/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "sys.h"

#define TEST_KEYWORD_AS "--this-is_a_TEST-KeYWord"
#define TEST_ARG_KEYWORD "my-option"
#define TEST_ARG_ARGV_STRING "--my-option=my-value"
#define TEST_CHECKSUM_STR_DATA "abc"
#define TEST_CHECKSUM_STR_RESULT 0x2c17398c
#define TEST_CHECKSUM_INT32_DATA 1234
#define TEST_CHECKSUM_INT32_RESULT 0x5d3d512e
#define TEST_CHECKSUM_INT64_DATA 0x1234567890abcdefLL
#define TEST_CHECKSUM_INT64_RESULT 0x5fb58579
#define COLOR_ASCII "#12345678"
#define COLOR_8_R 0x12
#define COLOR_8_G 0x34
#define COLOR_8_B 0x56
#define COLOR_8_A 0x78
#ifdef LW6_MS_WINDOWS
#define PATH_WITH_SLASH "\\foo\\bar\\"
#define PATH_WITHOUT_SLASH "\\foo\\bar"
#define PATH_CONCAT "\\foo\\bar\\foo\\bar"
#define PATH_CONCAT_PARENT "\\foo\\bar\\foo"
#define PATH_RELATIVE "..\\..\\foo\\bar"
#define PATH_RELATIVE_UNPARENT "..\\foo\\bar"
#else
#define PATH_WITH_SLASH "/foo/bar/"
#define PATH_WITHOUT_SLASH "/foo/bar"
#define PATH_CONCAT "/foo/bar/foo/bar"
#define PATH_CONCAT_PARENT "/foo/bar/foo"
#define PATH_RELATIVE "../../foo/bar"
#define PATH_RELATIVE_UNPARENT "../foo/bar"
#endif
#define TEST_PATH_CWD "./"
#define TEST_PATH_SPLIT "/foo/\\bar\\"
#define MALLOC_SIZE 1048576
#define THREAD_N 5
#define THREAD_SLEEP_MAIN 0.66f
#define THREAD_SLEEP_CALLBACK 0.2f
#define THREAD_TEXT "foo bar"
#define THREAD_STRESS_DURATION 3.0f
#define TEST_SLEEP_TIME 3
#define TEST_SLEEP_TIME_SHORT_STEP 0.001f
#define BLANK_STR "\t "
#define ASCII_STR "foo,bar"
#define TEST_ARGC 4
#define TEST_ARGV0 "test.out"
#ifdef LW6_MS_WINDOWS
#define TEST_ARGV1 "--prefix=C:\\toto"
#define TEST_ARGV2 "-map-dir=C:\\home\\user\\my-maps\\"
#else
#define TEST_ARGV1 "--prefix=/toto"
#define TEST_ARGV2 "-map-dir=/home/user/my-maps/"
#endif
#define TEST_ARGV3 "THISisAbug!==--foobar=123---?\\"
#define TEST_FILENAME "test.txt"
#define TEST_CONTENT "This is a test"
#define TEST_DUMP "This...\n...is a dump\n"
#define TEST_HEXA_INT64 1
#define TEST_HEXA_INT32 123456789
#define TEST_HEXA_INT16 12345
#define TEST_HEXA_INT8 123
#define TEST_HEXA_FLOAT 2.0f
#define TEST_HEXA_STR1 "abc"
#define TEST_HEXA_STR2 "defghi"
#define TEST_HEXA_WHD {1,2,3}
#define TEST_HEXA_XYZ {4,5,6}
#define TEST_HEXA_COLOR {25,50,75,100}
#define TEST_KEYWORD LW6DEF_PREFIX
#ifdef LW6_MS_WINDOWS
#define TEST_ENV_CONCAT1 "\\my\\path\\1"
#define TEST_ENV_CONCAT2 "\\my\\path\\2"
#else
#define TEST_ENV_CONCAT1 "/my/path/1"
#define TEST_ENV_CONCAT2 "/my/path/2"
#endif
#define TEST_ENV_SETENV_KEYWORD "LW6_MY_KEY"
#define TEST_ENV_SETENV_VALUE "my-value"
#define TEST_ENV_KEYWORD "my-key"
#ifdef LW6_MS_WINDOWS
#define TEST_ENV_SPLIT "foo1\\foo2;foo3:bar1;bar2\\bar3"
#else
#define TEST_ENV_SPLIT "foo1/foo2;foo3:bar1;bar2/bar3"
#endif
#define TEST_REFORMAT_STR1 "this is a short string"
#define TEST_REFORMAT_STR2 "this is a very long string it should be cut in some places but wherever there is a very-very-very-very-very-very-very-very-very-very-very-very-very-very-very-very-very-very long word it won't be cut.\n\nIt should even handle newlines in some places but well, this is not absolutely necessary"
#define TEST_REFORMAT_PREFIX "# "
#define TEST_REFORMAT_COLUMNS 40
#define TEST_SPLIT_STR "/foo/bar"
#define TEST_SPLIT_CHAR 'o'
#define TEST_SERIALIZE 12345
#define TEST_SORT_LENGTH 10
#define TEST_SORT_INT {9,3,4,5,6,8,7,2,0,1};
#define TEST_SORT_INT_MIN 0
#define TEST_SORT_INT_MAX 9
#define TEST_SORT_FLOAT {9.0,3.0,4.0,5.0,6.0,8.0,7.0,2.0,0.0,1.0};
#define TEST_SORT_FLOAT_MIN 0.0
#define TEST_SORT_FLOAT_MAX 9.0
#define TEST_SORT_STR {"9","3","4","5","6","8","7","2","0","1"};
#define TEST_SORT_STR_MIN "0"
#define TEST_SORT_STR_MAX "9"
#define TEST_CONVERT_INT 421
#define TEST_CONVERT_BOOL 1
#define TEST_CONVERT_FLOAT 3.14f
#define TEST_RANDOM_RANGE 1000
#define TEST_RANDOM_FLOAT_MIN 100.0f
#define TEST_RANDOM_FLOAT_MAX 101.0f
#define TEST_PROGRESS_N 7
#define TEST_HISTORY_MSG1 "1 too old"
#define TEST_HISTORY_MSG2 "2 too old 2"
#define TEST_HISTORY_MSG3 "3 oldest"
#define TEST_HISTORY_MSG4 "4 message this is long and should be truncated message this is long and should be truncated message this is long and should be truncated message this is long and should be truncated message this is long and should be truncated message this is long and should be truncated"
#define TEST_HISTORY_MSG5 "5 most recent"
#define TEST_HISTORY_SLEEP 5.0
#define TEST_HISTORY_TIMEOUT 2
#define TEST_SHAPE_W 640
#define TEST_SHAPE_H 480
#define TEST_SHAPE_D 3
#define TEST_SHAPE_X 123
#define TEST_SHAPE_Y 456
#define TEST_SHAPE_Z 1
#define TEST_SHAPE_MIN_WH 10
#define TEST_SHAPE_MAX_WH 1000
#define TEST_SHAPE_MAX_D 5

/*
 * Testing functions in arg.c
 */
static int
test_arg ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *value = NULL;
    int argc = TEST_ARGC;
    char *argv[TEST_ARGC] =
      { TEST_ARGV0, TEST_ARGV1, TEST_ARGV2, TEST_ARGV3 };

    lw6sys_log (LW6SYS_LOG_NOTICE, _("trying to match \"%s\""),
		TEST_ARG_ARGV_STRING);
    if (lw6sys_arg_match (TEST_ARG_KEYWORD, TEST_ARG_ARGV_STRING))
      {
	// does match!
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _
		    ("command line arg \"%s\" does not match keyword \"%s\""),
		    TEST_ARG_ARGV_STRING, TEST_ARG_KEYWORD);
	ret = 0;
      }

    if (lw6sys_arg_exists (argc, argv, TEST_KEYWORD))
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("keyword \"%s\" is present"),
		    TEST_KEYWORD);
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING, _("keyword \"%s\" not present"),
		    TEST_KEYWORD);
	ret = 0;
      }

    value = lw6sys_arg_get_value (argc, argv, TEST_KEYWORD);
    if (value)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("value for \"%s\" is \"%s\""),
		    TEST_KEYWORD, value);
	LW6SYS_FREE (value);
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING, _("no value for keyword \"%s\""),
		    TEST_KEYWORD);
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Used to test lw6sys_assoc_map.
 */
static void
assoc_map_func (void *func_data, char *key, void *value)
{
  int *ret = (int *) func_data;
  int *i = (int *) value;

  lw6sys_log (LW6SYS_LOG_NOTICE,
	      _("assoc item ret=%d, key=\"%s\", value=%d"), *ret, key, *i);
}

/*
 * Testing functions in assoc.c
 */
static int
test_assoc ()
{
  int ret = 1;
  lw6sys_assoc_t *assoc;
  lw6sys_assoc_t *assoc_copy;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("new/delete on assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_free (assoc);
  }

  {
    int a = 3, b = 5, c = 7;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("set/unset/has_key/get/map on int assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_set (&assoc, "a", (void *) &a);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    lw6sys_assoc_set (&assoc, "c", (void *) &a);
    lw6sys_assoc_set (&assoc, "c", (void *) &c);
    lw6sys_assoc_unset (assoc, "b");
    lw6sys_assoc_map (assoc, &assoc_map_func, &ret);
    lw6sys_assoc_sort_and_map (assoc, &assoc_map_func, &ret);
    ret = ret && lw6sys_assoc_has_key (assoc, "a");
    ret = ret && !lw6sys_assoc_has_key (assoc, "b");
    ret = ret && !lw6sys_assoc_has_key (assoc, "this key does not exist");
    ret = ret && (*((int *) lw6sys_assoc_get (assoc, "a")) == 3);
    ret = ret && ((int *) lw6sys_assoc_get (assoc, "b") == NULL);
    ret = ret && (*((int *) lw6sys_assoc_get (assoc, "c")) == 7);
    lw6sys_assoc_free (assoc);
  }

  {
    char *str1, *str2, *str3;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("testing free_func callback and dup on string assoc"));
    assoc = lw6sys_assoc_new (&lw6sys_free_callback);
    str1 = LW6SYS_MALLOC (5);
    strncpy (str1, "str1", 5);
    lw6sys_assoc_set (&assoc, "1", (void *) str1);
    str2 = LW6SYS_MALLOC (5);
    strncpy (str2, "str2", 5);
    lw6sys_assoc_set (&assoc, "2", (void *) str2);
    str3 = LW6SYS_MALLOC (5);
    strncpy (str3, "str3", 5);
    lw6sys_assoc_set (&assoc, "3", (void *) str3);
    lw6sys_assoc_unset (assoc, "1");
    lw6sys_assoc_unset (assoc, "3");
    assoc_copy =
      lw6sys_assoc_dup (assoc, (lw6sys_dup_func_t) lw6sys_str_copy);
    ret = ret && !lw6sys_assoc_has_key (assoc_copy, "1");
    ret = ret && lw6sys_assoc_has_key (assoc_copy, "2");
    ret = ret && !lw6sys_assoc_has_key (assoc_copy, "3");
    ret = ret
      && strcmp ((char *) lw6sys_assoc_get (assoc_copy, "2"), "str2") == 0;
    lw6sys_assoc_free (assoc_copy);
    lw6sys_assoc_free (assoc);
  }

  {
    int a = 3, b = 5;
    lw6sys_list_t *keys;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing keys on int assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_set (&assoc, "a", (void *) &a);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    keys = lw6sys_assoc_keys (assoc);
    lw6sys_assoc_free (assoc);
    ret = ret && lw6sys_list_length (keys) == 2;
    lw6sys_list_free (keys);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Test functions in build.c
 */
static int
test_build ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("build version is \"%s\""),
		lw6sys_build_get_version ());

    lw6sys_build_log_all ();
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in checksum.c
 */
static int
test_checksum ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    u_int32_t checksum = 0;

    checksum = lw6sys_checksum_str (TEST_CHECKSUM_STR_DATA);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("checksum for \"%s\" is %08x (should be %08x)"),
		TEST_CHECKSUM_STR_DATA, checksum, TEST_CHECKSUM_STR_RESULT);
    if (checksum != TEST_CHECKSUM_STR_RESULT)
      {
	ret = 0;
      }

    checksum = lw6sys_checksum_int32 (TEST_CHECKSUM_INT32_DATA);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("checksum for %d is %08x (should be %08x)"),
		TEST_CHECKSUM_INT32_DATA, checksum,
		TEST_CHECKSUM_INT32_RESULT);
    if (checksum != TEST_CHECKSUM_INT32_RESULT)
      {
	ret = 0;
      }

    checksum = lw6sys_checksum_int64 (TEST_CHECKSUM_INT64_DATA);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("checksum for %lld is %08x (should be %08x)"),
		TEST_CHECKSUM_INT64_DATA, checksum,
		TEST_CHECKSUM_INT64_RESULT);
    if (checksum != TEST_CHECKSUM_INT64_RESULT)
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static void
rgb_to_hsv_to_rgb (char *label, lw6sys_color_8_t color_8)
{
  lw6sys_color_hsv_t color_hsv;
  lw6sys_color_8_t color_8_back;

  lw6sys_log (LW6SYS_LOG_NOTICE, _("%s init r=%d,g=%d,b=%d"), label,
	      color_8.r, color_8.g, color_8.b);
  lw6sys_color_rgb_to_hsv (&color_hsv, color_8);
  lw6sys_log (LW6SYS_LOG_NOTICE, _("%s HSV convert h=%.2f,s=%.2f,v=%.2f"),
	      label, color_hsv.h, color_hsv.s, color_hsv.v);
  color_8_back = lw6sys_color_hsv_to_rgb (&color_hsv);
  lw6sys_log (LW6SYS_LOG_NOTICE, _("%s RGB convert r=%d,g=%d,b=%d"), label,
	      color_8_back.r, color_8_back.g, color_8_back.b);
}

/*
 * Testing functions in color.c
 */
static int
test_color ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_color_8_t color_8;
    lw6sys_color_f_t color_f;
    lw6sys_color_8_t color_8_average[3];
    float distance;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("interpreting color \"%s\""), COLOR_ASCII);
    color_8 = lw6sys_color_a_to_8 (COLOR_ASCII);
    if (color_8.r != COLOR_8_R || color_8.g != COLOR_8_G
	|| color_8.b != COLOR_8_B || color_8.a != COLOR_8_A)
      {
	ret = 0;
      }

    lw6sys_color_8_to_f (&color_f, color_8);
    color_8 = lw6sys_color_f_to_8 (&color_f);

    rgb_to_hsv_to_rgb ("white", LW6SYS_COLOR_8_WHITE);
    rgb_to_hsv_to_rgb ("black", LW6SYS_COLOR_8_BLACK);
    rgb_to_hsv_to_rgb ("red", LW6SYS_COLOR_8_RED);
    rgb_to_hsv_to_rgb ("green", LW6SYS_COLOR_8_GREEN);
    rgb_to_hsv_to_rgb ("blue", LW6SYS_COLOR_8_BLUE);
    rgb_to_hsv_to_rgb (LW6DEF_TEST, color_8);

    color_8 =
      lw6sys_color_ponderate (LW6SYS_COLOR_8_RED, LW6SYS_COLOR_8_BLUE, 0.5f);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("color between red & blue is r=%d,g=%d,b=%d"), color_8.r,
		color_8.g, color_8.b);

    color_8_average[0] = LW6SYS_COLOR_8_RED;
    color_8_average[1] = LW6SYS_COLOR_8_RED;
    color_8_average[2] = LW6SYS_COLOR_8_GREEN;
    color_8 = lw6sys_color_average (3, color_8_average);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("color average of red+red+green is r=%d,g=%d,b=%d"),
		color_8.r, color_8.g, color_8.b);
    distance =
      lw6sys_color_distance (LW6SYS_COLOR_8_GREEN, LW6SYS_COLOR_8_BLUE);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("distance between green and blue is %.2f"), distance);

    lw6sys_color_8_solid (&color_8);
    lw6sys_color_f_solid (&color_f);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in convert.c
 */
static int
test_convert ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *str;
    int i;
    float f;

    str = lw6sys_itoa (TEST_CONVERT_INT);
    if (str)
      {
	i = lw6sys_atoi (str);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("int=%d -> str=\"%s\" -> int=%d"),
		    TEST_CONVERT_INT, str, i);
	if (i != TEST_CONVERT_INT)
	  {
	    ret = 0;
	  }
	LW6SYS_FREE (str);
      }
    else
      {
	ret = 0;
      }

    str = lw6sys_btoa (TEST_CONVERT_BOOL);
    if (str)
      {
	i = lw6sys_atob (str);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("bool=%d -> str=\"%s\" -> bool=%d"),
		    TEST_CONVERT_BOOL, str, i);
	if (i != TEST_CONVERT_BOOL)
	  {
	    ret = 0;
	  }
	LW6SYS_FREE (str);
      }
    else
      {
	ret = 0;
      }

    str = lw6sys_ftoa (TEST_CONVERT_FLOAT);
    if (str)
      {
	f = lw6sys_atof (str);
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _("float=%f -> str=\"%s\" -> float=%f"),
		    TEST_CONVERT_FLOAT, str, f);
	if (f != TEST_CONVERT_FLOAT)
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_
			("%f!=%f: while this is not 100% blocking, floating point conversion on this computer is suspicious"),
			TEST_CONVERT_FLOAT, f);
	  }
	LW6SYS_FREE (str);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in dump.c
 */
static int
test_dump ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *user_dir = NULL;

    user_dir = lw6sys_get_default_user_dir ();
    if (user_dir)
      {
	if (!lw6sys_dir_exists (user_dir))
	  {
	    lw6sys_create_dir (user_dir);
	  }
	lw6sys_dump_clear (user_dir);
	if (!lw6sys_dump (user_dir, TEST_DUMP))
	  {
	    ret = 0;
	  }
	LW6SYS_FREE (user_dir);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in env.c
 */
static int
test_env ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *value;
    char *concat;

    concat = lw6sys_env_concat (TEST_ENV_CONCAT1, TEST_ENV_CONCAT2);
    if (concat)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("ENV concat is \"%s\""), concat);
	LW6SYS_FREE (concat);
      }
    else
      {
	ret = 0;
      }
    if (!lw6sys_setenv (TEST_ENV_SETENV_KEYWORD, TEST_ENV_SETENV_VALUE))
      {
	lw6sys_log (LW6SYS_LOG_WARNING, _("unable to set ENV \"%s\""),
		    TEST_ENV_SETENV_KEYWORD);
	ret = 0;
      }
    if (!lw6sys_env_exists (TEST_ENV_KEYWORD))
      {
	lw6sys_log (LW6SYS_LOG_WARNING, _("ENV \"%s\" doesn't exist"),
		    TEST_ENV_KEYWORD);
	ret = 0;
      }
    value = lw6sys_getenv (TEST_ENV_KEYWORD);
    if (value)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("ENV value for \"%s\" is \"%s\""),
		    TEST_ENV_KEYWORD, value);
	LW6SYS_FREE (value);
      }
    else
      {
	ret = 0;
      }
    value = lw6sys_get_username ();
    if (value)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("username=\"%s\""), value);
	LW6SYS_FREE (value);
      }
    else
      {
	ret = 0;
      }
    value = lw6sys_get_hostname ();
    if (value)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("hostname=\"%s\""), value);
	LW6SYS_FREE (value);
      }
    else
      {
	ret = 0;
      }
  }

  {
    lw6sys_list_t *splitted = NULL;
    char *data = NULL;

    splitted = lw6sys_env_split (TEST_ENV_SPLIT);
    if (splitted)
      {
	lw6sys_list_free (splitted);
      }
    else
      {
	ret = 0;
      }

    splitted = lw6sys_env_split (TEST_ENV_SPLIT);
    if (splitted)
      {
	while (splitted
	       && (data = (char *) lw6sys_lifo_pop (&splitted)) != NULL)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("splitted env \"%s\" contains \"%s\""),
			TEST_ENV_SPLIT, data);
	    LW6SYS_FREE (data);
	  }
	//lw6sys_list_free(splitted);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in file.c
 */
static int
test_file ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *content = NULL;
    char *user_dir = NULL;
    char *filename = NULL;
    int argc = TEST_ARGC;
    char *argv[TEST_ARGC] =
      { TEST_ARGV0, TEST_ARGV1, TEST_ARGV2, TEST_ARGV3 };

    user_dir = lw6sys_get_user_dir (argc, argv);
    if (user_dir)
      {
	if (!lw6sys_dir_exists (user_dir))
	  {
	    lw6sys_create_dir (user_dir);
	  }
	filename = lw6sys_path_concat (user_dir, TEST_FILENAME);
	if (filename)
	  {
	    if (lw6sys_write_file_content (filename, TEST_CONTENT))
	      {
		content = lw6sys_read_file_content (filename);

		if (content && !strcmp (TEST_CONTENT, content))
		  {
		    lw6sys_log (LW6SYS_LOG_NOTICE,
				_("file \"%s\" contains \"%s\""),
				filename, content);
		    LW6SYS_FREE (content);
		  }
		else
		  {
		    ret = 0;
		  }
	      }
	    else
	      {
		ret = 0;
	      }
	    LW6SYS_FREE (filename);
	  }
	LW6SYS_FREE (user_dir);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in hash.c
 */
static int
test_hash ()
{
  int ret = 1;
  lw6sys_hash_t *hash;
  lw6sys_hash_t *hash_copy;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("new/delete on hash"));
    hash = lw6sys_hash_new (NULL, 7);
    lw6sys_hash_free (hash);
  }

  {
    int a = 3, b = 5, c = 7;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("set/unset/has_key/get/map on int hash"));
    hash = lw6sys_hash_new (NULL, 7);
    lw6sys_hash_set (hash, "a", (void *) &a);
    lw6sys_hash_set (hash, "b", (void *) &b);
    lw6sys_hash_set (hash, "c", (void *) &a);
    lw6sys_hash_set (hash, "c", (void *) &c);
    lw6sys_hash_unset (hash, "b");
    lw6sys_hash_map (hash, &assoc_map_func, &ret);
    lw6sys_hash_sort_and_map (hash, &assoc_map_func, &ret);
    ret = ret && lw6sys_hash_has_key (hash, "a");
    ret = ret && !lw6sys_hash_has_key (hash, "b");
    ret = ret && !lw6sys_hash_has_key (hash, "this key does not exist");
    ret = ret && (*((int *) lw6sys_hash_get (hash, "a")) == 3);
    ret = ret && ((int *) lw6sys_hash_get (hash, "b") == NULL);
    ret = ret && (*((int *) lw6sys_hash_get (hash, "c")) == 7);
    lw6sys_hash_free (hash);
  }

  {
    char *str1, *str2, *str3;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("testing free_func callback and dup on string hash"));
    hash = lw6sys_hash_new (&lw6sys_free_callback, 7);
    str1 = LW6SYS_MALLOC (5);
    strncpy (str1, "str1", 5);
    lw6sys_hash_set (hash, "1", (void *) str1);
    str2 = LW6SYS_MALLOC (5);
    strncpy (str2, "str2", 5);
    lw6sys_hash_set (hash, "2", (void *) str2);
    str3 = LW6SYS_MALLOC (5);
    strncpy (str3, "str3", 5);
    lw6sys_hash_set (hash, "3", (void *) str3);
    lw6sys_hash_unset (hash, "1");
    lw6sys_hash_unset (hash, "3");
    hash_copy = lw6sys_hash_dup (hash, (lw6sys_dup_func_t) lw6sys_str_copy);
    ret = ret && !lw6sys_hash_has_key (hash_copy, "1");
    ret = ret && lw6sys_hash_has_key (hash_copy, "2");
    ret = ret && !lw6sys_hash_has_key (hash_copy, "3");
    ret = ret
      && strcmp ((char *) lw6sys_hash_get (hash_copy, "2"), "str2") == 0;
    lw6sys_hash_free (hash_copy);
    lw6sys_hash_free (hash);
  }

  {
    int a = 3, b = 5;
    lw6sys_list_t *keys;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing keys on int hash"));
    hash = lw6sys_hash_new (NULL, 7);
    lw6sys_hash_set (hash, "a", (void *) &a);
    lw6sys_hash_set (hash, "b", (void *) &b);
    lw6sys_hash_set (hash, "b", (void *) &b);
    keys = lw6sys_hash_keys (hash);
    lw6sys_hash_free (hash);
    ret = ret && lw6sys_list_length (keys) == 2;
    lw6sys_list_free (keys);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_hexa ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_hexa_serializer_t *hexa_serializer;
    char *hexa_string = NULL;
    int64_t test_64;
    int32_t test_32;
    int16_t test_16;
    int8_t test_8;
    float test_float;
    char *test_str = NULL;
    lw6sys_whd_t test_whd = TEST_HEXA_WHD;
    lw6sys_xyz_t test_xyz = TEST_HEXA_XYZ;
    lw6sys_color_8_t test_color = TEST_HEXA_COLOR;

    hexa_serializer = lw6sys_hexa_serializer_new ("");
    if (hexa_serializer)
      {
	lw6sys_hexa_serializer_rewind (hexa_serializer);
	ret = ret
	  && lw6sys_hexa_serializer_push_int64 (hexa_serializer,
						TEST_HEXA_INT64);
	ret = ret
	  && lw6sys_hexa_serializer_push_int32 (hexa_serializer,
						TEST_HEXA_INT32);
	ret = ret
	  && lw6sys_hexa_serializer_push_int16 (hexa_serializer,
						TEST_HEXA_INT16);
	ret = ret
	  && lw6sys_hexa_serializer_push_int8 (hexa_serializer,
					       TEST_HEXA_INT8);
	ret = ret
	  && lw6sys_hexa_serializer_push_float (hexa_serializer,
						TEST_HEXA_FLOAT);
	ret = ret
	  && lw6sys_hexa_serializer_push_str (hexa_serializer,
					      TEST_HEXA_STR1);
	ret = ret
	  && lw6sys_hexa_serializer_push_str (hexa_serializer,
					      TEST_HEXA_STR2);
	ret = ret
	  && lw6sys_hexa_serializer_push_whd (hexa_serializer, test_whd);
	ret = ret
	  && lw6sys_hexa_serializer_push_xyz (hexa_serializer, test_xyz);
	ret = ret
	  && lw6sys_hexa_serializer_push_color (hexa_serializer, test_color);
	if (ret)
	  {
	    hexa_string = lw6sys_hexa_serializer_as_string (hexa_serializer);
	    if (hexa_string)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("hexa string is \"%s\""),
			    hexa_string);
		LW6SYS_FREE (hexa_string);
	      }

	    lw6sys_hexa_serializer_rewind (hexa_serializer);

	    if (lw6sys_hexa_serializer_eof (hexa_serializer))
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _("unexpected serializer EOF"));
		ret = 0;
	      }

	    if (lw6sys_hexa_serializer_pop_int64 (hexa_serializer, &test_64))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("int64 loaded, value=%d"),
			    (int) test_64);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_int32 (hexa_serializer, &test_32))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("int32 loaded, value=%d"),
			    (int) test_32);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_int16 (hexa_serializer, &test_16))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("int16 loaded, value=%d"),
			    (int) test_16);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_int8 (hexa_serializer, &test_8))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("int8 loaded, value=%d"),
			    (int) test_8);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_float
		(hexa_serializer, &test_float))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("float loaded, value=%f"),
			    test_float);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_str (hexa_serializer, &test_str))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("str loaded, value=\"%s\""),
			    test_str);
		LW6SYS_FREE (test_str);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_str (hexa_serializer, &test_str))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("str loaded, value=\"%s\""),
			    test_str);
		LW6SYS_FREE (test_str);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_whd (hexa_serializer, &test_whd))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("whd loaded, value=%dx%dx%d"), (int) test_whd.w,
			    (int) test_whd.h, (int) test_whd.d);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_xyz (hexa_serializer, &test_xyz))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("xyz loaded, value=%d,%d,%d"), (int) test_xyz.x,
			    (int) test_xyz.y, (int) test_xyz.z);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_pop_color
		(hexa_serializer, &test_color))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("color loaded, value=#%02x%02x%02x%02x"),
			    (int) test_color.r, (int) test_color.g,
			    (int) test_color.b, (int) test_color.a);
	      }
	    else
	      {
		ret = 0;
	      }
	    if (lw6sys_hexa_serializer_eof (hexa_serializer))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("serializer EOF"));
	      }
	    else
	      {
		ret = 0;
	      }
	  }

	lw6sys_hexa_serializer_free (hexa_serializer);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in history.c
 */
static int
test_history ()
{
  int ret = 0;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char **history;
    char **i;

    lw6sys_history_init ();
    lw6sys_history_register (TEST_HISTORY_MSG1);
    lw6sys_history_register (TEST_HISTORY_MSG2);
    lw6sys_sleep (TEST_HISTORY_SLEEP);
    lw6sys_history_register (TEST_HISTORY_MSG3);
    lw6sys_history_register (TEST_HISTORY_MSG4);
    lw6sys_history_register (TEST_HISTORY_MSG5);
    history = lw6sys_history_get (TEST_HISTORY_TIMEOUT);
    if (history)
      {
	for (i = history; i[0]; ++i)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("history msg \"%s\""), i[0]);
	    ret = 1;
	  }
	lw6sys_history_free (history);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in i18n.c
 */
static int
test_i18n ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *utf8_str;
    int local_ret = 0;

    utf8_str = lw6sys_locale_to_utf8 (ASCII_STR);
    if (utf8_str)
      {
	if (strcmp (utf8_str, ASCII_STR) == 0)
	  {
	    local_ret = 1;
	  }

	LW6SYS_FREE (utf8_str);
      }

    ret = ret && local_ret;
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in id.c
 */
static int
test_id ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    u_int16_t id_16 = 0;
    u_int32_t id_32 = 0;
    u_int64_t id_64 = 0;
    char *id_str = NULL;
    u_int64_t id_long = 0;

    id_16 = lw6sys_generate_id_16 ();
    id_str = lw6sys_id_ltoa (id_16);
    if (id_str)
      {
	id_long = lw6sys_id_atol (id_str);
	if (id_long == id_16)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("here's a random 16-bit id \"%s\""), id_str);
	  }
	else
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_("erreur converting id, src=%" LW6SYS_PRINTF_LL
			  "x, dst=%" LW6SYS_PRINTF_LL "x"), (u_int64_t) id_16,
			id_long);
	    ret = 0;
	  }
	LW6SYS_FREE (id_str);
      }

    id_32 = lw6sys_generate_id_32 ();
    id_str = lw6sys_id_ltoa (id_32);
    if (id_str)
      {
	id_long = lw6sys_id_atol (id_str);
	if (id_long == id_32)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("here's a random 32-bit id \"%s\""), id_str);
	  }
	else
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_("erreur converting id, src=%" LW6SYS_PRINTF_LL
			  "x, dst=%" LW6SYS_PRINTF_LL "x"), (u_int64_t) id_32,
			id_long);
	    ret = 0;
	  }
	LW6SYS_FREE (id_str);
      }

    id_64 = lw6sys_generate_id_64 ();
    id_str = lw6sys_id_ltoa (id_64);
    if (id_str)
      {
	id_long = lw6sys_id_atol (id_str);
	if (id_long == id_64)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("here's a random 64-bit id \"%s\""), id_str);
	  }
	else
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_("erreur converting id, src=%" LW6SYS_PRINTF_LL
			  "x, dst=%" LW6SYS_PRINTF_LL "x"), (u_int64_t) id_64,
			id_long);
	    ret = 0;
	  }
	LW6SYS_FREE (id_str);
      }

  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in keyword.c
 */
static int
test_keyword ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *str;

    str = lw6sys_keyword_as_key (TEST_KEYWORD_AS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("keyword as key is \"%s\""), str);
	LW6SYS_FREE (str);
      }
    str = lw6sys_keyword_as_arg (TEST_KEYWORD_AS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("keyword as arg is \"%s\""), str);
	LW6SYS_FREE (str);
      }
    str = lw6sys_keyword_as_env (TEST_KEYWORD_AS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("keyword as env is \"%s\""), str);
	LW6SYS_FREE (str);
      }
    str = lw6sys_keyword_as_xml (TEST_KEYWORD_AS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("keyword as xml is \"%s\""), str);
	LW6SYS_FREE (str);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Used to test lw6sys_list_map.
 */
static void
list_map_func (void *func_data, void *data)
{
  int *ret = (int *) func_data;
  int *i = (int *) data;

  lw6sys_log (LW6SYS_LOG_NOTICE, _("list item ret=%d, data=%d"), *ret, *i);
}

/*
 * Testing functions in list.c
 */
static int
test_list ()
{
  int ret = 1;
  lw6sys_list_t *list;
  lw6sys_list_t *list_copy;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("new/delete/is_empty on list"));
    list = lw6sys_list_new (NULL);
    ret = ret && lw6sys_list_is_empty (list);
    lw6sys_list_free (list);
  }

  {
    int a = 3, b = 5, c = 7;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("lifo push/pop on int list"));
    list = lw6sys_list_new (NULL);
    lw6sys_lifo_push (&list, (void *) &a);
    lw6sys_lifo_push (&list, (void *) &b);
    lw6sys_lifo_push (&list, (void *) &c);
    lw6sys_list_map (list, &list_map_func, &ret);
    ret = ret && lw6sys_list_length (list) == 3;
    ret = ret && (*((int *) lw6sys_lifo_pop (&list)) == c);
    ret = ret && (*((int *) lw6sys_lifo_pop (&list)) == b);
    ret = ret && (*((int *) lw6sys_lifo_pop (&list)) == a);
    lw6sys_list_free (list);
  }

  {
    int a = 2, b = 4, c = 6, d = 8;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("fifo push/pop on int list"));
    list = lw6sys_list_new (NULL);
    lw6sys_fifo_push (&list, (void *) &a);
    lw6sys_fifo_push (&list, (void *) &b);
    lw6sys_fifo_push (&list, (void *) &c);
    lw6sys_fifo_push (&list, (void *) &d);
    lw6sys_list_map (list, &list_map_func, &ret);
    ret = ret && lw6sys_list_length (list) == 4;
    ret = ret && (*((int *) lw6sys_fifo_pop (&list)) == a);
    ret = ret && (*((int *) lw6sys_fifo_pop (&list)) == b);
    ret = ret && (*((int *) lw6sys_fifo_pop (&list)) == c);
    ret = ret && (*((int *) lw6sys_fifo_pop (&list)) == d);
    lw6sys_list_free (list);
  }

  {
    int a = 32, b = 64, c = 128, d = 256, e = 512;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("front/back push/pop on int list"));
    list = lw6sys_list_new (NULL);
    lw6sys_list_push_front (&list, (void *) &a);
    lw6sys_list_push_front (&list, (void *) &b);
    lw6sys_list_push_back (&list, (void *) &c);
    lw6sys_list_push_back (&list, (void *) &d);
    lw6sys_list_push_back (&list, (void *) &e);
    lw6sys_list_map (list, &list_map_func, &ret);
    ret = ret && lw6sys_list_length (list) == 5;
    ret = ret && (*((int *) lw6sys_list_pop_back (&list)) == e);
    ret = ret && (*((int *) lw6sys_list_pop_back (&list)) == d);
    ret = ret && (*((int *) lw6sys_list_pop_front (&list)) == b);
    ret = ret && (*((int *) lw6sys_list_pop_front (&list)) == a);
    ret = ret && (*((int *) lw6sys_list_pop_front (&list)) == c);
    lw6sys_list_free (list);
  }

  {
    char *str1, *str2, *str3, *str4;

    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("testing free_func callback and dup on string list"));
    list = lw6sys_list_new (&lw6sys_free_callback);
    str1 = lw6sys_str_copy ("str1");
    lw6sys_lifo_push (&list, (void *) str1);
    str2 = lw6sys_str_copy ("str2");
    lw6sys_fifo_push (&list, (void *) str2);
    str3 = lw6sys_str_copy ("str3");
    lw6sys_lifo_push (&list, (void *) str3);
    str4 = lw6sys_str_copy ("str4");
    lw6sys_fifo_push (&list, (void *) str4);
    list_copy = lw6sys_list_dup (list, (lw6sys_dup_func_t) lw6sys_str_copy);
    ret = ret && lw6sys_list_length (list_copy) == 4;
    lw6sys_list_free (list_copy);
    lw6sys_list_free (list);
  }

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing final pop behavior"));

    list = lw6sys_list_new (NULL);
    if (list)
      {
	lw6sys_list_pop_front (&list);
	if (list)
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_("final pop front didn't free list"));
	    lw6sys_list_free (list);
	    ret = 0;
	  }
      }

    list = lw6sys_list_new (NULL);
    if (list)
      {
	lw6sys_list_pop_back (&list);
	if (list)
	  {
	    lw6sys_log (LW6SYS_LOG_WARNING,
			_("final pop back didn't free list"));
	    lw6sys_list_free (list);
	    ret = 0;
	  }
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in log.c
 */
static int
test_log ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    int log_level;

    log_level = lw6sys_log_get_level ();
    lw6sys_log_set_level (log_level);
    /*
     * We do not test WARNING & ERROR to avoid confusion...
     */
    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing log_notice %s=%d"),
		"log_level=%d", log_level);
    lw6sys_log (LW6SYS_LOG_INFO, _("testing log_info %s=%d"), "log_level=%d",
		log_level);
    // normally, this one does now show with defaults settings...
    lw6sys_log (LW6SYS_LOG_DEBUG, _("testing log_debug %s=%d"),
		"log_level=%d", log_level);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in mem.c
 */
static int
test_mem ()
{
  int ret = 1;
  char *ptr = NULL;
  LW6SYS_TEST_FUNCTION_BEGIN;

  lw6sys_log (LW6SYS_LOG_NOTICE,
	      _("testing MALLOC/FREE (%d bytes)"), MALLOC_SIZE);
  ptr = (char *) LW6SYS_MALLOC (MALLOC_SIZE * sizeof (char));
  ret = ret && (ptr != NULL);
  LW6SYS_FREE (ptr);
  // same tests without using macros, fixes link when LW6_OPTIMZE is defined
  ptr = (char *) lw6sys_malloc (MALLOC_SIZE, __FILE__, __LINE__);
  if (ptr)
    {
      lw6sys_free (ptr, __FILE__, __LINE__);
    }
  ptr = (char *) lw6sys_calloc (MALLOC_SIZE, __FILE__, __LINE__);
  if (ptr)
    {
      ptr = lw6sys_realloc (ptr, MALLOC_SIZE + 1, __FILE__, __LINE__);
      if (ptr)
	{
	  lw6sys_free (ptr, __FILE__, __LINE__);
	}
    }
  lw6sys_log (LW6SYS_LOG_NOTICE, _("%d megs"), lw6sys_megabytes_available ());
  if (lw6sys_is_big_endian ())
    {
      lw6sys_log (LW6SYS_LOG_NOTICE, _("big endian"));
    }
  if (lw6sys_is_little_endian ())
    {
      lw6sys_log (LW6SYS_LOG_NOTICE, _("little endian"));
    }

  ret = ret && lw6sys_check_types_size ();

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in mutex.c
 */
static int
test_mutex ()
{
  int ret = 0;
  void *mutex = NULL;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing mutex functions"));
    mutex = lw6sys_mutex_create ();
    if (mutex)
      {
	if (lw6sys_mutex_lock (mutex))
	  {
	    if (lw6sys_mutex_trylock (mutex))
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _
			    ("strange, trylock on locked mutex should return 0"));
	      }
	    else
	      {
		if (lw6sys_mutex_unlock (mutex))
		  {
		    ret = 1;
		  }
	      }
	  }
	lw6sys_mutex_destroy (mutex);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing options functions
 */
static int
test_options ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *path;

    path = lw6sys_get_default_user_dir ();
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("default user dir is \"%s\""), path);
	LW6SYS_FREE (path);
      }
    else
      {
	ret = 0;
      }

    path = lw6sys_get_default_prefix ();
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("default prefix is \"%s\""), path);
	LW6SYS_FREE (path);
      }
    else
      {
	ret = 0;
      }

    lw6sys_options_log_defaults ();
  }

  {
    char *path;
    int argc = TEST_ARGC;
    char *argv[TEST_ARGC] =
      { TEST_ARGV0, TEST_ARGV1, TEST_ARGV2, TEST_ARGV3 };

    path = lw6sys_get_user_dir (argc, argv);
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("user dir is \"%s\""), path);
	LW6SYS_FREE (path);
      }
    else
      {
	ret = 0;
      }

    path = lw6sys_get_prefix (argc, argv);
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("prefix is \"%s\""), path);
	LW6SYS_FREE (path);
      }
    else
      {
	ret = 0;
      }

    lw6sys_options_log (argc, argv);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in nop.c
 */
static int
test_nop ()
{
  int ret = 0;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    ret = lw6sys_true () && !lw6sys_false ();
  }

  LW6SYS_TEST_FUNCTION_END return ret;
}

/*
 * Used to test lw6sys_path_split
 */
static void
path_split_func (void *func_data, void *data)
{
  char *str = (char *) data;

  lw6sys_log (LW6SYS_LOG_NOTICE, _("path part \"%s\""), str);
}

/*
 * Testing functions in path.c
 */
static int
test_path ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *path = NULL;
    int relative = 0;
    int cwd = 0;
    lw6sys_list_t *list = NULL;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("add/strip slash"));
    path = lw6sys_path_add_slash (PATH_WITHOUT_SLASH);
    if (path)
      {
	ret = ret && (strcmp (path, PATH_WITH_SLASH) == 0);
	LW6SYS_FREE (path);
      }
    path = lw6sys_path_strip_slash (PATH_WITH_SLASH);
    if (path)
      {
	ret = ret && (strcmp (path, PATH_WITHOUT_SLASH) == 0);
	LW6SYS_FREE (path);
      }
    lw6sys_log (LW6SYS_LOG_NOTICE, _("concat path"));
    path = lw6sys_path_concat (PATH_WITH_SLASH, PATH_WITH_SLASH);
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("concatenated path = \"%s\""), path);
	ret = ret && (strcmp (path, PATH_CONCAT) == 0);
	LW6SYS_FREE (path);
      }
    relative = lw6sys_path_is_relative (PATH_RELATIVE);
    if (relative)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _("path \"%s\" is relative"), PATH_RELATIVE);
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _("path \"%s\" not considered relative"), PATH_RELATIVE);
	ret = 0;
      }
    cwd = lw6sys_path_is_cwd (TEST_PATH_CWD);
    if (cwd)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _("path \"%s\" is cwd"), TEST_PATH_CWD);
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _("path \"%s\" not considered being cwd"), TEST_PATH_CWD);
	ret = 0;
      }
    path = lw6sys_path_parent (PATH_CONCAT);
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("parent path = \"%s\""), path);
	ret = ret && (strcmp (path, PATH_CONCAT_PARENT) == 0);
	LW6SYS_FREE (path);
      }
    path = lw6sys_path_unparent (PATH_RELATIVE);
    if (path)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("unparent path = \"%s\""), path);
	ret = ret && (strcmp (path, PATH_RELATIVE_UNPARENT) == 0);
	LW6SYS_FREE (path);
      }
    path = lw6sys_path_unparent_no_malloc (PATH_RELATIVE);
    lw6sys_log (LW6SYS_LOG_NOTICE, _("unparent path = \"%s\""), path);
    ret = ret && (strcmp (path, PATH_RELATIVE_UNPARENT) == 0);

    list = lw6sys_path_split (TEST_PATH_SPLIT);
    lw6sys_log (LW6SYS_LOG_NOTICE, _("splitting \"%s\""), TEST_PATH_SPLIT);
    if (list)
      {
	lw6sys_list_map (list, path_split_func, NULL);
	lw6sys_list_free (list);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in progress.c
 */
static int
test_progress ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_progress_t progress;
    lw6sys_progress_t progress1;
    lw6sys_progress_t progress2;
    lw6sys_progress_t progress3;
    lw6sys_progress_t progress4;
    float value = 0.0f;
    int i = 0;

    lw6sys_progress_default (&progress, &value);
    for (i = 0; i < TEST_PROGRESS_N; ++i)
      {
	lw6sys_progress_update (&progress, 0, TEST_PROGRESS_N, i);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("progress=%0.2f"),
		    *(progress.value));
      }
    lw6sys_progress_begin (&progress);
    lw6sys_progress_half (&progress);
    lw6sys_progress_end (&progress);
    lw6sys_progress_split (&progress1, &progress2, &progress);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("progress1 %0.2f/%0.2f progress2 %0.2f/%0.2f"),
		progress1.min, progress1.max, progress2.min, progress2.max);
    lw6sys_progress_split3 (&progress1, &progress2, &progress3, &progress);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_
		("progress1 %0.2f/%0.2f progress2 %0.2f/%0.2f progress3 %0.2f/%0.2f"),
		progress1.min, progress1.max, progress2.min, progress2.max,
		progress3.min, progress3.max);
    lw6sys_progress_split4 (&progress1, &progress2, &progress3, &progress4,
			    &progress);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_
		("progress1 %0.2f/%0.2f progress2 %0.2f/%0.2f progress3 %0.2f/%0.2f progress4 %0.2f/%0.2f"),
		progress1.min, progress1.max, progress2.min, progress2.max,
		progress3.min, progress3.max, progress4.min, progress4.max);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in random.c
 */
static int
test_random ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    int i = 0;
    float f = 0.0f;

    i = lw6sys_random (TEST_RANDOM_RANGE);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("here's a random number between 0 and %d: %d"),
		TEST_RANDOM_RANGE, i);
    f = lw6sys_random_float (TEST_RANDOM_FLOAT_MIN, TEST_RANDOM_FLOAT_MAX);
    lw6sys_log (LW6SYS_LOG_NOTICE,
		_("here's a random float number between %f and %f: %f"),
		TEST_RANDOM_FLOAT_MIN, TEST_RANDOM_FLOAT_MAX, f);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in sort.c
 */
static int
test_sort ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_list_t *list = NULL;
    int array[TEST_SORT_LENGTH] = TEST_SORT_INT;
    int i;

    list = lw6sys_list_new (NULL);
    if (list)
      {
	for (i = 0; i < TEST_SORT_LENGTH; ++i)
	  {
	    lw6sys_lifo_push (&list, (void *) (array + i));
	  }
	lw6sys_sort (&list, lw6sys_sort_int_callback);
	if (list)
	  {
	    if ((*((int *) (list->data))) == TEST_SORT_INT_MIN)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("min sorted value is %d"),
			    TEST_SORT_INT_MIN);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _("min sorted value is %d and should be %d"),
			    *((int *) (list->data)), TEST_SORT_INT_MIN);
		ret = 0;
	      }
	  }
	lw6sys_sort (&list, lw6sys_sort_int_desc_callback);
	if (list)
	  {
	    if ((*((int *) (list->data))) == TEST_SORT_INT_MAX)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("max sorted value is %d"),
			    TEST_SORT_INT_MAX);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _("max sorted value is %d and should be %d"),
			    *((int *) (list->data)), TEST_SORT_INT_MAX);
		ret = 0;
	      }
	  }
	if (list)
	  {
	    lw6sys_list_free (list);
	  }
	else
	  {
	    ret = 0;
	  }
      }
    else
      {
	ret = 0;
      }
  }

  {
    lw6sys_list_t *list = NULL;
    float array[TEST_SORT_LENGTH] = TEST_SORT_FLOAT;
    int i;

    list = lw6sys_list_new (NULL);
    if (list)
      {
	for (i = 0; i < TEST_SORT_LENGTH; ++i)
	  {
	    lw6sys_lifo_push (&list, (void *) (array + i));
	  }
	lw6sys_sort (&list, lw6sys_sort_float_callback);
	if (list)
	  {
	    if ((*((float *) (list->data))) == TEST_SORT_FLOAT_MIN)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("min sorted value is %f"),
			    TEST_SORT_FLOAT_MIN);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _("min sorted value is %f and should be %f"),
			    *((float *) (list->data)), TEST_SORT_FLOAT_MIN);
		ret = 0;
	      }
	  }
	lw6sys_sort (&list, lw6sys_sort_float_desc_callback);
	if (list)
	  {
	    if ((*((float *) (list->data))) == TEST_SORT_FLOAT_MAX)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("max sorted value is %f"),
			    TEST_SORT_FLOAT_MAX);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _("max sorted value is %f and should be %f"),
			    *((float *) (list->data)), TEST_SORT_FLOAT_MAX);
		ret = 0;
	      }
	  }
	if (list)
	  {
	    lw6sys_list_free (list);
	  }
	else
	  {
	    ret = 0;
	  }
      }
    else
      {
	ret = 0;
      }
  }

  {
    lw6sys_list_t *list = NULL;
    char *array[TEST_SORT_LENGTH] = TEST_SORT_STR;
    int i;

    list = lw6sys_list_new (NULL);
    if (list)
      {
	for (i = 0; i < TEST_SORT_LENGTH; ++i)
	  {
	    lw6sys_lifo_push (&list, (void *) array[i]);
	  }
	lw6sys_sort (&list, lw6sys_sort_str_callback);
	if (list)
	  {
	    if (!strcmp ((char *) (list->data), TEST_SORT_STR_MIN))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("min sorted value is \"%s\""),
			    TEST_SORT_STR_MIN);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _
			    ("min sorted value is \"%s\" and should be \"%s\""),
			    (char *) (list->data), TEST_SORT_STR_MIN);
		ret = 0;
	      }
	  }
	lw6sys_sort (&list, lw6sys_sort_str_desc_callback);
	if (list)
	  {
	    if (!strcmp ((char *) (list->data), TEST_SORT_STR_MAX))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("max sorted value is \"%s\""),
			    TEST_SORT_STR_MAX);
	      }
	    else
	      {
		lw6sys_log (LW6SYS_LOG_WARNING,
			    _
			    ("max sorted value is \"%s\" and should be \"%s\""),
			    (char *) (list->data), TEST_SORT_STR_MAX);
		ret = 0;
	      }
	  }
	if (list)
	  {
	    lw6sys_list_free (list);
	  }
	else
	  {
	    ret = 0;
	  }
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Used to test lw6sys_split...
 */
static void
str_split_func (void *func_data, void *data)
{
  char *str = (char *) data;

  lw6sys_log (LW6SYS_LOG_NOTICE, _("split part \"%s\""), str);
}

/*
 * Testing functions in str.c
 */
static int
test_str ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    char *str;
    lw6sys_list_t *list;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("testing is_blank on \"%s\""),
		BLANK_STR);
    ret = ret && lw6sys_str_is_blank (BLANK_STR);
    str =
      lw6sys_str_reformat (TEST_REFORMAT_STR1, TEST_REFORMAT_PREFIX,
			   TEST_REFORMAT_COLUMNS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("reformatted string is \"%s\""),
		    str);
	LW6SYS_FREE (str);
      }
    else
      {
	ret = 0;
      }

    str =
      lw6sys_str_reformat (TEST_REFORMAT_STR2, TEST_REFORMAT_PREFIX,
			   TEST_REFORMAT_COLUMNS);
    if (str)
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("reformatted string is \"%s\""),
		    str);
	LW6SYS_FREE (str);
      }
    else
      {
	ret = 0;
      }

    list = lw6sys_str_split (TEST_SPLIT_STR, TEST_SPLIT_CHAR);
    if (list)
      {
	lw6sys_list_map (list, str_split_func, NULL);
	lw6sys_list_free (list);
      }

    list = lw6sys_str_split_no_0 (TEST_SPLIT_STR, TEST_SPLIT_CHAR);
    if (list)
      {
	lw6sys_list_map (list, str_split_func, NULL);
	lw6sys_list_free (list);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in sdl.c
 */
static int
test_sdl ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_NOTICE, _("register/unregister SDL"));
    ret = ret && lw6sys_sdl_register () && lw6sys_sdl_unregister ();
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in serial.c
 */
static int
test_serial ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    int64_t test64 = 0;
    int32_t test32 = 0;
    int16_t test16 = 0;
    unsigned char buffer[LW6SYS_SIZEOF_INT64];

    lw6sys_serialize_int64 (buffer, TEST_SERIALIZE);
    test64 = lw6sys_unserialize_int64 (buffer);
    if (test64 != TEST_SERIALIZE)
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _("serialize failed on 64-bit integer (%d/%d)"),
		    (int) test64, TEST_SERIALIZE);
	ret = 0;
      }

    lw6sys_serialize_int32 (buffer, TEST_SERIALIZE);
    test32 = lw6sys_unserialize_int32 (buffer);
    if (test32 != TEST_SERIALIZE)
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _("serialize failed on 32-bit integer (%d/%d)"),
		    (int) test32, TEST_SERIALIZE);
	ret = 0;
      }

    lw6sys_serialize_int16 (buffer, TEST_SERIALIZE);
    test16 = lw6sys_unserialize_int16 (buffer);
    if (test16 != TEST_SERIALIZE)
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _("serialize failed on 16-bit integer (%d/%d)"),
		    (int) test16, TEST_SERIALIZE);
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in shape.c
 */
static int
test_shape ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6sys_whd_t shape = { TEST_SHAPE_W, TEST_SHAPE_H, TEST_SHAPE_D };
    lw6sys_xyz_t pos = { TEST_SHAPE_X, TEST_SHAPE_Y, TEST_SHAPE_Z };

    if (lw6sys_shape_check_min_max_whd
	(&shape, TEST_SHAPE_MIN_WH, TEST_SHAPE_MAX_WH, TEST_SHAPE_MAX_D))
      {
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _("shape %dx%dx%d respects [%d-%d-%d] constraints"),
		    (int) shape.w, (int) shape.h, (int) shape.d,
		    TEST_SHAPE_MIN_WH, TEST_SHAPE_MAX_WH, TEST_SHAPE_MAX_D);
      }
    else
      {
	ret = 0;
      }
    if (lw6sys_shape_check_pos (&shape, &pos))
      {
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _("pos %d,%d,%d is inside shape %dx%dx%d"), (int) pos.x,
		    (int) pos.y, (int) pos.z, (int) shape.w, (int) shape.h,
		    (int) shape.d);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static void
thread_func (void *callback_data)
{
  int i = 0;
  char *text = NULL;

  text = (char *) callback_data;
  lw6sys_log (LW6SYS_LOG_NOTICE, _("thread_callback text=\"%s\""), text);
  for (i = 0; i < THREAD_N; ++i)
    {
      lw6sys_log (LW6SYS_LOG_NOTICE, _("thread_callback step %d"), i + 1);
      lw6sys_sleep (THREAD_SLEEP_CALLBACK);
    }
}

static void
thread_join (void *callback_data)
{
  LW6SYS_FREE (callback_data);
}

typedef struct thread_stress_data_s
{
  int run;
  int count;
  int ret;
} thread_stress_data_t;

static void
thread_stress_func (void *callback_data)
{
  thread_stress_data_t *thread_stress_data = NULL;
  void *child;

  thread_stress_data = (thread_stress_data_t *) callback_data;

  while (thread_stress_data->run)
    {
      child = lw6sys_thread_create (NULL, NULL, NULL, 0);
      if (child)
	{
	  lw6sys_thread_join (child);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, _("unable to create child"));
	  thread_stress_data->ret = 0;
	}
      thread_stress_data->count++;
    }
}

/*
 * Testing functions in thread.c
 */
static int
test_thread ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    void *thread_handler = NULL;
    char *text = lw6sys_str_copy (THREAD_TEXT);
    int i = 0;
    thread_stress_data_t thread_stress_data;

    ret = 1;
    if (text)
      {
	thread_handler =
	  lw6sys_thread_create (&thread_func, &thread_join, (void *) text, 0);
	if (thread_handler)
	  {
	    for (i = 0; i < THREAD_N; ++i)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("thread_main step %d, callback_done=%d"), i + 1,
			    lw6sys_thread_is_callback_done (thread_handler));
		lw6sys_sleep (THREAD_SLEEP_MAIN);
	      }

	    lw6sys_thread_join (thread_handler);
	  }
	else
	  {
	    ret = 0;
	  }
      }
    else
      {
	ret = 0;
      }
    if (ret)
      {
	thread_stress_data.run = 1;
	thread_stress_data.count = 0;
	thread_stress_data.ret = 1;
	thread_handler =
	  lw6sys_thread_create (&thread_stress_func, NULL,
				&thread_stress_data, 0);
	if (thread_handler)
	  {
	    lw6sys_sleep (THREAD_STRESS_DURATION);
	    thread_stress_data.run = 0;
	    lw6sys_thread_join (thread_handler);
	    ret = ret && thread_stress_data.ret;
	    if (ret)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("%d threads (%0.1f/sec)"),
			    thread_stress_data.count,
			    (float) (((float) thread_stress_data.count) /
				     THREAD_STRESS_DURATION));
	      }
	  }
	else
	  {
	    ret = 0;
	  }
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing functions in time.c
 */
static int
test_time ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    int i = 0;
    int nb_steps = 0;
    int last_uptime;

    lw6sys_log (LW6SYS_LOG_NOTICE, _("timestamp %lld"), lw6sys_timestamp ());
    lw6sys_log (LW6SYS_LOG_NOTICE, _("uptime %d"), lw6sys_uptime ());
    last_uptime = lw6sys_uptime ();
    lw6sys_log (LW6SYS_LOG_NOTICE, _("sleep %d seconds"), TEST_SLEEP_TIME);
    lw6sys_sleep (TEST_SLEEP_TIME);
    lw6sys_log (LW6SYS_LOG_NOTICE, _("lasted %d"),
		lw6sys_uptime () - last_uptime);
    last_uptime = lw6sys_uptime ();
    nb_steps = TEST_SLEEP_TIME / TEST_SLEEP_TIME_SHORT_STEP;
    lw6sys_log (LW6SYS_LOG_NOTICE, _("sleep %d seconds using %d %0.6f steps"),
		TEST_SLEEP_TIME, nb_steps, TEST_SLEEP_TIME_SHORT_STEP);
    for (i = 0; i < nb_steps; ++i)
      {
	lw6sys_sleep (TEST_SLEEP_TIME_SHORT_STEP);
      }
    lw6sys_log (LW6SYS_LOG_NOTICE, _("lasted %d"),
		lw6sys_uptime () - last_uptime);
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/**
 * lw6sys_test
 *
 * Runs the @sys module test suite, testing most (if not all...)
 * functions. Note that some tests perform file system operations
 * and might therefore fail on a read-only filesystem, or if
 * user permissions are not sufficient.
 *
 * Return value: 1 if test is successfull, 0 on error.
 */
int
lw6sys_test ()
{
  int ret = 0;

  ret = test_arg () && test_assoc () && test_build () && test_checksum ()
    && test_color () && test_convert () && test_dump () && test_env ()
    && test_file () && test_hash () && test_hexa () && test_history ()
    && test_i18n () && test_id () && test_keyword () && test_list ()
    && test_log () && test_mem () && test_mutex () && test_options ()
    && test_nop () && test_path () && test_progress () && test_random ()
    && test_sdl () && test_serial () && test_shape () && test_sort ()
    && test_str () && test_thread () && test_time ();

  return ret;
}
