/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "map.h"

void
lw6map_body_defaults (lw6map_body_t * body, int w, int h, int d)
{
  int layer;

  lw6map_body_clear (body);
  body->shape.w = w;
  body->shape.h = h;
  body->shape.d = d;
  if (!lw6sys_shape_check_min_max_whd
      (&body->shape, LW6MAP_MIN_SIDE, LW6MAP_MAX_SIDE, LW6MAP_MAX_DEPTH))
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("incorrect default map size %dx%dx%d"), body->shape.w,
		  body->shape.h, body->shape.d);
    }
  for (layer = 0; layer < body->shape.d; ++layer)
    {
      lw6map_layer_defaults (&(body->layers[layer]), w, h);
    }
}

void
lw6map_body_clear (lw6map_body_t * body)
{
  int layer;

  for (layer = 0; layer < LW6MAP_MAX_DEPTH; ++layer)
    {
      lw6map_layer_clear (&(body->layers[layer]));
    }

  memset (body, 0, sizeof (lw6map_body_t));
}

static int
find_first_free_point (lw6map_body_t * body, int *found_x,
		       int *found_y, int *found_z)
{
  int ret = 0;
  int x, y, z;

  *found_x = 0;
  *found_y = 0;
  *found_z = 0;
  for (y = 0; y < body->shape.h && !ret; ++y)
    {
      for (x = 0; x < body->shape.w && !ret; ++x)
	{
	  for (z = 0; z < body->shape.d; ++z)
	    {
	      if (lw6map_body_get (body, x, y, z) > 0)
		{
		  ret = 1;
		  *found_x = x;
		  *found_y = y;
		  *found_z = z;
		}
	    }
	}
    }

  return ret;
}

/*
 * Updates a body point if needed, returns true if value was changed
 */
static int
update_if_needed (lw6map_body_t * dst, lw6map_body_t * src, int x,
		  int y, int z)
{
  int ret = 0;

  if (lw6map_body_get (dst, x, y, z) != lw6map_body_get (src, x, y, z))
    {
      lw6map_body_set (dst, x, y, z, lw6map_body_get (src, x, y, z));
      ret = 1;
    }

  return ret;
}

/*
 * Checks the maps for "holes", that is fight zones that wouldn't
 * be connected with the main area, and fills them with walls.
 */
int
lw6map_body_check_and_fix_holes (lw6map_body_t * body, lw6map_rules_t * rules)
{
  int ret = 0;
  lw6map_body_t fixed_body;
  int layer;
  int x;
  int y;
  int z;
  int tx;
  int ty;
  int tz;

  memset (&fixed_body, 0, sizeof (lw6map_body_t));
  fixed_body.shape = body->shape;
  ret = 1;
  for (layer = 0; layer < fixed_body.shape.d; ++layer)
    {
      fixed_body.layers[layer].shape = fixed_body.shape;
      fixed_body.layers[layer].data =
	(unsigned char *) LW6SYS_CALLOC (body->shape.w * body->shape.h *
					 sizeof (unsigned char));
      if (!fixed_body.layers[layer].data)
	{
	  ret = 0;
	}
    }

  if (ret)
    {
      int found;

      /*
       * We first set one point, which will spread all over.
       */
      ret = find_first_free_point (body, &x, &y, &z)
	&& update_if_needed (&fixed_body, body, x, y, z);
      found = ret ? 1 : 0;

      while (found)
	{
	  found = 0;

	  for (y = 0; y < body->shape.h; ++y)
	    {
	      for (x = 0; x < body->shape.w; ++x)
		{
		  for (z = 0; z < body->shape.d; ++z)
		    {
		      if (lw6map_body_get (&fixed_body, x, y, z) > 0)
			{
			  tx = x + 1;
			  ty = y;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x + 1;
			  ty = y + 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x;
			  ty = y + 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x - 1;
			  ty = y + 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tz = z + 1;
			  lw6map_coords_fix_z (rules, &body->shape, &tz);
			  found +=
			    update_if_needed (&fixed_body, body, x, y, tz);
			}
		    }
		}
	    }

	  for (y = body->shape.h - 1; y >= 0; --y)
	    {
	      for (x = body->shape.w - 1; x >= 0; --x)
		{
		  for (z = 0; z < body->shape.d; ++z)
		    {
		      if (lw6map_body_get (&fixed_body, x, y, z) > 0)
			{
			  tx = x - 1;
			  ty = y;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x - 1;
			  ty = y - 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x;
			  ty = y - 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tx = x + 1;
			  ty = y - 1;
			  lw6map_coords_fix_xy (rules, &body->shape, &tx,
						&ty);
			  found +=
			    update_if_needed (&fixed_body, body, tx, ty, z);
			  tz = z - 1;
			  lw6map_coords_fix_z (rules, &body->shape, &tz);
			  found +=
			    update_if_needed (&fixed_body, body, x, y, tz);
			}
		    }


		}
	    }
	}
    }

  if (ret)
    {
      for (layer = 0; layer < fixed_body.shape.d; ++layer)
	{
	  LW6SYS_FREE (body->layers[layer].data);
	  body->layers[layer].data = fixed_body.layers[layer].data;
	}
    }
  else
    {
      for (layer = 0; layer < fixed_body.shape.d; ++layer)
	{
	  if (fixed_body.layers[layer].data)
	    {
	      LW6SYS_FREE (fixed_body.layers[layer].data);
	    }
	}
    }

  return ret;
}

int
lw6map_body_coord_from_texture (lw6map_level_t * level, int *body_x,
				int *body_y, int texture_x, int texture_y)
{
  int ret = 0;

  if (level->texture.w > 0 && level->texture.h > 0)
    {
      (*body_x) = (texture_x * level->body.shape.w) / level->texture.w;
      (*body_y) = (texture_y * level->body.shape.h) / level->texture.h;
      (*body_x) =
	lw6sys_max (0, lw6sys_min (level->body.shape.w - 1, *body_x));
      (*body_y) =
	lw6sys_max (0, lw6sys_min (level->body.shape.h - 1, *body_y));
      ret = 1;
    }
  else
    {
      (*body_x) = 0;
      (*body_y) = 0;
    }

  return ret;
}


u_int8_t
lw6map_body_get_with_texture_coord (lw6map_level_t * level,
				    int texture_x, int texture_y, int z)
{
  unsigned char ret = 0;
  int body_x;
  int body_y;

  if (lw6map_body_coord_from_texture
      (level, &body_x, &body_y, texture_x, texture_y))
    {
      ret = lw6map_body_get (&level->body, body_x, body_y, z);
    }

  return ret;
}
