/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>

#include "config.h"
#include "sys.h"

/*
 * Tests if a file exists and is readable.
 */
int
lw6sys_file_exists (char *filename)
{
  int ret = 0;
  FILE *f;

  f = fopen (filename, "r");
  if (f)
    {
      ret = 1;
      fclose (f);
    }

  return ret;
}

/*
 * Tests if a directory exists.
 */
int
lw6sys_dir_exists (char *dirname)
{
  struct dirent **namelist;
  int n;
  int ret = 0;

  n = scandir (dirname, &namelist, NULL, NULL);
  if (n >= 0)
    {
      ret = 1;
      free (namelist);
    }

  return ret;
}

/*
 * Wrapper on mkdir with all the options we like, + logs.
 */
int
lw6sys_create_dir (char *dirname)
{
  int ret = 0;

  if (mkdir (dirname, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP) == 0)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "sys", _("creating dir \"%s\""), dirname);
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("unable to create dir \"%s\""),
		  dirname);
    }

  return ret;
}

/*
 * Creates all the directories need to store a file,
 * that is, if the file is /a/b/c/d.txt, then this
 * function will try to create /a /a/b and /a/b/c.
 * It's a bit like calling "install -d `dirname $FILE`".
 */
int
lw6sys_create_dir_for_file (char *filename)
{
  char *left;
  char *tmp;
  int ret = 1;

  left = filename;
  if (left[0] == '/')
    {
      left++;
    }
  while ((tmp = strchr (left, '/')) != NULL)
    {
      int length = tmp - filename;
      char *buf;

      left = tmp;

      buf = LW6SYS_MALLOC (length + 1);
      if (buf)
	{
	  memcpy (buf, filename, length);
	  buf[length] = '\0';
	  if (!lw6sys_dir_exists (buf))
	    {
	      ret = ret && lw6sys_create_dir (buf);
	    }
	  LW6SYS_FREE (buf);
	}

      left++;
    }

  return ret;
}

/*
 * Returns a string containing a path to a data file,
 * the idea is to try file then data/file then ../data/file
 * then /usr/local/share/liquidwar6/data/file to ease
 * up the developpement of the game (make it possible to
 * use ./ data when developping) while using the "real"
 * data in /usr/local in production mode.
 * Returns a newly allocated string if file exists, or
 * NULL if it does not exists anywhere.
 */
char *
lw6sys_find_data_file (char *filename)
{
  char *real_filename = NULL;
  char *test_filename = NULL;
  char *prefixes[] = {
    "",
    "data/",
    "../data/",
    LW6_DATA_DIR,
    LW6_TOP_SRC_DIR "data/",
    NULL
  };
  char **prefix = NULL;
  char *last_prefix = NULL;

  for (prefix = prefixes, last_prefix = prefixes[0]; *prefix != NULL;
       ++prefix)
    {
      if (!real_filename)
	{
	  test_filename = lw6sys_str_concat (*prefix, filename);
	  if (test_filename)
	    {
	      if (lw6sys_file_exists (test_filename))
		{
		  real_filename = test_filename;
		}
	      else
		{
		  last_prefix = *prefix;
		  LW6SYS_FREE (test_filename);
		}
	    }
	}
    }

  if (!real_filename)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("data file \"%s%s\" does not exists"), LW6_DATA_DIR,
		  filename);
    }

  return real_filename;
}
