/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <time.h>
#include <string.h>

#include "config.h"
#include "sys.h"

#define LOG_FILE "log.txt"
#define CTIME_BUF_SIZE 30	// more than 26 is enough
#define CTIME_ZERO 24

char *
lw6sys_log_get_default_file ()
{
  char *dir = NULL;
  char *ret;

  dir = getenv ("HOME");
  if (!dir)
    {
      dir = ".";
    }

  ret = lw6sys_str_concat (dir, "/." PACKAGE_TARNAME "/" LOG_FILE);

  return ret;
}

void
lw6sys_log_clear ()
{
  FILE *f = NULL;
  char *filename;

  filename = lw6sys_log_get_default_file ();
  if (filename)
    {
      lw6sys_create_dir_for_file (filename);
      f = fopen (filename, "w");
      if (f)
	{
	  fclose (f);
	}
      LW6SYS_FREE (filename);
    }
}

static FILE *
open_default_file ()
{
  FILE *ret = NULL;
  char *name;

  name = lw6sys_log_get_default_file ();
  if (name)
    {
      ret = fopen (name, "a");
      LW6SYS_FREE (name);
    }

  return ret;
}

static void
log_to_file (FILE * f, char *level_str, char *file, int line, char *module,
	     char *fmt, va_list ap)
{
  char ctime_buf[CTIME_BUF_SIZE];
  time_t t;
  char *file_only;

  file_only = strrchr (file, '/');
  if (file_only && *file_only)
    {
      file_only++;
    }
  else
    {
      file_only = file;
    }

  t = time (NULL);
  ctime_r (&t, ctime_buf);
  ctime_buf[CTIME_ZERO] = '\0';

  fprintf (f, "%s\t%s%s\t%s:%d\t%s", ctime_buf, PACKAGE_TARNAME, module,
	   file_only, line, level_str);
  vfprintf (f, fmt, ap);
  fprintf (f, "\n");
  fflush (f);
}

static void
log_to_console (FILE * f, char *level_str, char *module,
		char *fmt, va_list ap)
{
  fprintf (f, "%s%s: %s", PACKAGE_TARNAME, module, level_str);
  vfprintf (f, fmt, ap);
  fprintf (f, "\n");
  fflush (f);
}

/*
 * Logs a string
 */
void
lw6sys_log (int level_id, char *file, int line, char *module, char *fmt, ...)
{
  char *level_str = NULL;
  int syslog_priority = 0;
  FILE *f;
  va_list ap;
  va_list ap2;

  va_start (ap, fmt);

  if (module == NULL)
    {
      module = "";
    }

  switch (level_id)
    {
    case LW6SYS_LOG_INFO_ID:
      level_str = "";
      syslog_priority = LOG_INFO;
      break;
    case LW6SYS_LOG_NOTICE_ID:
      level_str = "";
      syslog_priority = LOG_NOTICE;
      break;
    case LW6SYS_LOG_WARNING_ID:
      level_str = _("WARNING! ");
      syslog_priority = LOG_WARNING;
      break;
    case LW6SYS_LOG_ERROR_ID:
      level_str = _("ERROR! ");
      syslog_priority = LOG_ERR;
      break;
    default:			// LW6SYS_LOG_DEBUG_ID
      level_str = _("[debug] ");
      syslog_priority = LOG_DEBUG;
      break;
    }

  if (fmt != NULL)
    {
      if (level_id != LW6SYS_LOG_INFO_ID)
	{
	  va_copy (ap2, ap);
	  log_to_console (stdout, level_str, module, fmt, ap2);
	  va_end (ap2);
	}
      if (level_id >= LW6SYS_LOG_NOTICE_ID)
	{
	  va_copy (ap2, ap);
	  vsyslog (syslog_priority, fmt, ap2);
	  va_end (ap2);
	}
      f = open_default_file ();
      if (f)
	{
	  va_copy (ap2, ap);
	  log_to_file (f, level_str, file, line, module, fmt, ap2);
	  va_end (ap2);
	  fclose (f);
	}
    }

  va_end (ap);
}
