/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "config.h"
#include "net.h"
#include "net-internal.h"

int
_lw6net_socket_init (_LW6NET_CONTEXT * net_context)
{
  int ret = 1;

  return ret;
}

void
_lw6net_socket_quit (_LW6NET_CONTEXT * net_context)
{
  if (net_context->socket_counters.open_counter <
      net_context->socket_counters.close_counter)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "net",
		  _
		  ("%d sockets opened, but %d closed, there's probably a bug"),
		  net_context->socket_counters.open_counter,
		  net_context->socket_counters.close_counter);
    }
  if (net_context->socket_counters.open_counter >
      net_context->socket_counters.close_counter)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "net",
		  _
		  ("%d sockets opened, but %d closed, the only acceptable explanation is that there's a detached thread or something, which was idle when program ended"),
		  net_context->socket_counters.open_counter,
		  net_context->socket_counters.close_counter);
    }
  if (net_context->socket_counters.close_counter ==
      net_context->socket_counters.open_counter)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "net",
		  _("%d sockets opened and closed"),
		  net_context->socket_counters.open_counter);
    }
}

int
lw6net_socket_listen (void *net_context, char *ip, int port)
{
  int sock = -1;
  int listening = 0;
  struct sockaddr_in name;
  int enable = 1;
  int backlog;

  sock = socket (AF_INET, SOCK_STREAM, 0);
  if (sock >= 0)
    {
      backlog = ((_LW6NET_CONTEXT *) net_context)->const_data.listen_backlog;
      setsockopt (sock, SOL_SOCKET, SO_REUSEADDR,
		  (char *) &enable, sizeof (int));
      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      if (inet_aton (ip, &name.sin_addr) != 0)
	{
	  name.sin_port = htons (port);
	  if (bind (sock, (struct sockaddr *) &name, sizeof name) >= 0)
	    {
	      if (listen (sock, backlog) >= 0)
		{
		  ((_LW6NET_CONTEXT *) net_context)->socket_counters.
		    open_counter++;
		  listening = 1;
		}
	    }
	}
    }

  if (sock >= 0 && !listening)
    {
      close (sock);
      sock = -1;
    }

  return sock;
}

/*
 * WARNING! Due to the usage of inet_ntoa, this is code is not
 * reentrant, therefore not thread safe. Not a big problem since
 * most of the time accept is used in the main thread of a server
 * process but... still, be carefull.
 */
int
lw6net_socket_accept (void *net_context, char **incoming_ip,
		      int *incoming_port, int listening_sock, float delay)
{
  int new_sock = -1;
  int accepted = 0;
  struct sockaddr_in name;
  socklen_t namelen = sizeof (struct sockaddr_in);
  fd_set read;
  struct timeval tv;
  int res;
  int enable = 1;
  int disable = 0;
  struct linger li;

  *incoming_ip = NULL;
  *incoming_port = 0;

  if (listening_sock >= 0)
    {
      FD_ZERO (&read);
      FD_SET (listening_sock, &read);
      tv.tv_sec = (int) delay;
      tv.tv_usec = (int) (delay / 1000000.0f);

      res = select (listening_sock + 1, &read, NULL, NULL, &tv);
      if (res >= 1)
	{
	  new_sock =
	    accept (listening_sock, (struct sockaddr *) &name, &namelen);
	  if (new_sock >= 0)
	    {
	      li.l_onoff = 0;
	      li.l_linger = 0;
	      setsockopt (new_sock, SOL_SOCKET, SO_KEEPALIVE,
			  (char *) &enable, sizeof (int));
	      setsockopt (new_sock, SOL_SOCKET, SO_OOBINLINE,
			  (char *) &disable, sizeof (int));
	      setsockopt (new_sock, SOL_SOCKET, SO_LINGER,
			  (char *) &li, sizeof (struct linger));

	      //fcntl (new_sock, F_SETFL, O_NONBLOCK, 0);

	      (*incoming_ip) = lw6sys_str_copy (inet_ntoa (name.sin_addr));
	      (*incoming_port) = (int) ntohs (name.sin_port);
	      if (*incoming_ip)
		{
		  ((_LW6NET_CONTEXT *) net_context)->socket_counters.
		    open_counter++;
		  accepted = 1;
		}
	    }
	}
    }

  if (new_sock >= 0 && !accepted)
    {
      if (*incoming_ip)
	{
	  LW6SYS_FREE (*incoming_ip);
	}
      (*incoming_ip) = NULL;
      (*incoming_port) = 0;
      close (new_sock);
      new_sock = -1;
    }

  return new_sock;
}

int
lw6net_socket_connect (void *net_context, char *ip, int port)
{
  int sock = -1;
  int connected = 0;
  struct sockaddr_in name;
  int enable = 1;
  int disable = 0;
  struct linger li;

  sock = socket (AF_INET, SOCK_STREAM, 0);
  if (sock >= 0)
    {
      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      name.sin_port = 0;
      if (bind (sock, (struct sockaddr *) &name, sizeof name) >= 0)
	{
	  name.sin_family = AF_INET;
	  if (inet_aton (ip, &name.sin_addr) != 0)
	    {
	      name.sin_port = htons (port);
	      if (connect (sock, (struct sockaddr *) &name, sizeof name) >= 0)
		{
		  /*
		   * Added this code copied/paste from accept.
		   * don't know if it's usefull
		   */
		  li.l_onoff = 0;
		  li.l_linger = 0;
		  setsockopt (sock, SOL_SOCKET, SO_KEEPALIVE,
			      (char *) &enable, sizeof (int));
		  setsockopt (sock, SOL_SOCKET, SO_OOBINLINE,
			      (char *) &disable, sizeof (int));
		  setsockopt (sock, SOL_SOCKET, SO_LINGER,
			      (char *) &li, sizeof (struct linger));

		  //fcntl (sock, F_SETFL, O_NONBLOCK, 0);

		  lw6sys_log (LW6SYS_LOG_INFO, "net", _("connected on %s:%d"),
			      ip, port);

		  ((_LW6NET_CONTEXT *) net_context)->socket_counters.
		    open_counter++;
		  connected = 1;
		}
	    }
	}
    }

  if (sock >= 0 && !connected)
    {
      close (sock);
      sock = -1;
    }

  return sock;
}

void
lw6net_socket_close (void *net_context, int sock)
{
  if (sock >= 0)
    {
      ((_LW6NET_CONTEXT *) net_context)->socket_counters.close_counter++;
      close (sock);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "net",
		  _("can't close negative socket %d"), sock);
    }
}

static void
async_connect_callback_func (void *callback_data)
{
  char *state = NULL;
  _LW6NET_SOCKET_ASYNC_CONNECT_DATA *async_connect_data;

  async_connect_data = (_LW6NET_SOCKET_ASYNC_CONNECT_DATA *) callback_data;

  async_connect_data->sock =
    lw6net_socket_connect ((void *) (async_connect_data->context),
			   async_connect_data->ip, async_connect_data->port);

  if (async_connect_data->sock >= 0)
    {
      state = _("connected");
    }
  else
    {
      state = _("connexion failed");
    }

  lw6sys_log (LW6SYS_LOG_INFO, "net",
	      _("async connect on %s:%d has terminated, %s"),
	      async_connect_data->ip, async_connect_data->port, state);
}

static void
async_connect_callback_join (void *callback_data)
{
  _LW6NET_SOCKET_ASYNC_CONNECT_DATA *async_connect_data;

  async_connect_data = (_LW6NET_SOCKET_ASYNC_CONNECT_DATA *) callback_data;

  /*
   * It's important to have this here, nowhere else but at the very end
   * of the connect thread can we close the socket, it ever "got connected"
   * after we waited "long enough".
   */
  if (async_connect_data->close && async_connect_data->sock >= 0)
    {
      lw6net_socket_close (async_connect_data->context,
			   async_connect_data->sock);
    }

  LW6SYS_FREE (async_connect_data);
}

void *
lw6net_socket_async_connect_init (void *net_context, char *ip, int port)
{
  void *ret = NULL;
  _LW6NET_SOCKET_ASYNC_CONNECT_DATA *async_connect_data = NULL;

  async_connect_data =
    LW6SYS_CALLOC (sizeof (_LW6NET_SOCKET_ASYNC_CONNECT_DATA));
  if (async_connect_data)
    {
      async_connect_data->context = (_LW6NET_CONTEXT *) net_context;
      strncpy (async_connect_data->ip, ip, _LW6NET_IP_SIZE - 1);
      async_connect_data->ip[_LW6NET_IP_SIZE - 1] = '\0';
      async_connect_data->port = port;
      async_connect_data->sock = -1;
      async_connect_data->close = 1;
      // all other values set to 0 (calloc)
      ret =
	lw6sys_thread_create (&async_connect_callback_func,
			      &async_connect_callback_join,
			      (void *) async_connect_data, 0);
      if (ret)
	{
	  // ok
	  _lw6net_thread_register ((_LW6NET_CONTEXT *) net_context, ret);
	}
      else
	{
	  LW6SYS_FREE (async_connect_data);
	}
    }

  return ret;
}

int
lw6net_socket_async_connect_get (void *net_context, int *sock, void *handler)
{
  int ret = 0;
  _LW6NET_SOCKET_ASYNC_CONNECT_DATA *async_connect_data;

  if (lw6sys_thread_is_callback_done (handler))
    {
      async_connect_data =
	(_LW6NET_SOCKET_ASYNC_CONNECT_DATA *)
	lw6sys_thread_get_data (handler);
      (*sock) = async_connect_data->sock;
      async_connect_data->close = 0;
      ret = 1;
    }

  return ret;
}

void
lw6net_socket_async_connect_exit (void *net_context, void *handler)
{
  /*
   * Unregister should automatically call
   * lw6sys_thread_join(handler); but the advantage is that
   * this way, at the end of the program, we can join all threads
   * together and garbage collect things "properly".
   */
  _lw6net_thread_unregister ((_LW6NET_CONTEXT *) net_context, handler);
}

int
lw6net_socket_send (void *net_context, int sock, char *buf, int len,
		    float delay, int loop)
{
  int ret = 0;
  fd_set write;
  struct timeval tv;
  int select_ret;
  int total_sent;
  int sent = 0;
  int chunk_size;

  if (sock >= 0)
    {
      ret = 1;

      chunk_size = ((_LW6NET_CONTEXT *) net_context)->const_data.chunk_size;
      total_sent = 0;
      while (total_sent != len && ret)
	{
	  FD_ZERO (&write);
	  FD_SET (sock, &write);
	  tv.tv_sec = (int) delay;
	  tv.tv_usec = (int) ((delay - tv.tv_sec) * 1000000.0f);

	  select_ret = select (sock + 1, NULL, &write, NULL, &tv);

	  switch (select_ret)
	    {
	    case -1:
	      if (errno != EINTR && errno != ENOBUFS)
		{
		  lw6sys_log (LW6SYS_LOG_WARNING, "net",
			      _
			      ("error sending data on socket %d (select error %d)"),
			      sock, errno);
		  ret = 0;
		}
	      break;
	    case 1:
	      if (FD_ISSET (sock, &write))
		{
		  sent = send (sock,
			       buf + total_sent,
			       lw6sys_min (len - total_sent, chunk_size), 0);
		  if (sent > 0 && sent <= len - total_sent)
		    {
		      total_sent += sent;
		    }
		  else
		    {
		      lw6sys_log (LW6SYS_LOG_INFO, "net",
				  _
				  ("can't send data on socket %d (%d bytes put)"),
				  sock, sent);
		      ret = 0;
		    }
		}
	      break;
	    default:
	      lw6sys_log (LW6SYS_LOG_INFO, "net",
			  _
			  ("can't send data on socket %d (select returned %d)"),
			  sock, select_ret);
	      ret = 0;
	    }

	  if ((!loop) && (total_sent != len))
	    {
	      lw6sys_log (LW6SYS_LOG_INFO, "net",
			  _("can't send data on socket %d (%d/%d)"), sock,
			  total_sent, len);
	      ret = 0;
	    }
	}
    }

  return ret;
}

int
lw6net_socket_peek (void *net_context, int sock, char *buf, int len,
		    float delay)
{
  fd_set read;
  struct timeval tv;
  int select_ret;
  char *buf2;
  int available = 0;

  if (sock >= 0)
    {
      if (buf)
	{
	  buf2 = buf;
	}
      else
	{
	  buf2 = (char *) LW6SYS_MALLOC (len);
	}
      if (buf2)
	{
	  FD_ZERO (&read);
	  FD_SET (sock, &read);
	  tv.tv_sec = (int) delay;
	  tv.tv_usec = (int) ((delay - tv.tv_sec) * 1000000.0f);

	  select_ret = select (sock + 1, &read, NULL, NULL, &tv);

	  if (select_ret > 0)
	    {
	      if (FD_ISSET (sock, &read))
		{
		  available = recv (sock, buf2, len, MSG_PEEK);
		}
	    }

	  if (!buf)
	    {
	      LW6SYS_FREE (buf2);
	    }
	}
    }

  if (available < 0)
    {
      available = 0;
    }

  return available;
}

int
lw6net_socket_recv (void *net_context, int sock, char *buf, int len,
		    float delay, int loop)
{
  int ret = 0;
  fd_set read;
  struct timeval tv;
  int select_ret;
  int total_received;
  int received;
  int chunk_size;

  if (sock >= 0)
    {
      ret = 1;
      chunk_size = ((_LW6NET_CONTEXT *) net_context)->const_data.chunk_size;
      memset (buf, 0, len);
      total_received = 0;
      while (total_received != len && ret)
	{
	  FD_ZERO (&read);
	  FD_SET (sock, &read);
	  tv.tv_sec = (int) delay;
	  tv.tv_usec = (int) ((delay - tv.tv_sec) * 1000000.0f);

	  select_ret = select (sock + 1, &read, NULL, NULL, &tv);

	  switch (select_ret)
	    {
	    case -1:
	      if (errno != EINTR)
		{
		  lw6sys_log (LW6SYS_LOG_WARNING, "net",
			      _
			      ("error receiving data on socket %d (select error %d)"),
			      sock, errno);
		  ret = 0;
		}
	      break;
	    case 1:
	      if (FD_ISSET (sock, &read))
		{
		  received = recv (sock,
				   buf + total_received,
				   lw6sys_min (len - total_received,
					       chunk_size), 0);
		  if (received > 0 && received <= len - total_received)
		    {
		      total_received += received;
		    }
		  else
		    {
		      lw6sys_log (LW6SYS_LOG_WARNING, "net",
				  _
				  ("can't recv data on socket %d (got %d bytes)"),
				  sock, received);
		      ret = 0;
		    }
		}
	      break;
	    default:
	      lw6sys_log (LW6SYS_LOG_WARNING, "net",
			  _
			  ("can't recv data on socket %d (select returned %d)"),
			  sock, select_ret);
	      ret = 0;
	    }

	  if ((!loop) && (total_received != len))
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING, "net",
			  _("can't recv data on socket %d (%d/%d)"), sock,
			  total_received, len);
	      ret = 0;
	    }
	}
    }

  return ret;
}

int
lw6net_socket_is_alive (void *net_context, int sock)
{
  int ret = 0;

  fd_set write;
  struct timeval tv;
  int select_ret;

  if (sock >= 0)
    {
      ret = 1;

      FD_ZERO (&write);
      FD_SET (sock, &write);
      tv.tv_sec = 0;
      tv.tv_usec = 0;

      select_ret = select (sock + 1, NULL, &write, NULL, &tv);

      switch (select_ret)
	{
	case -1:
	  if (errno != EINTR && errno != ENOBUFS)
	    {
	      // socket is closed...
	      ret = 0;
	    }
	  break;
	case 1:
	  if (!FD_ISSET (sock, &write))
	    {
	      // socket is closed
	      ret = 0;
	    }
	  break;
	default:
	  // socket is closed
	  ret = 0;
	}
    }

  return ret;
}
