/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <jpeglib.h>

#include "config.h"
#include "net.h"
#include "net-internal.h"

#define SCREENSHOT_FILE "screenshot.jpeg"

char *
lw6net_screenshot_get_default_jpeg ()
{
  char *dir = NULL;
  char *ret;

  dir = getenv ("HOME");
  if (!dir)
    {
      dir = ".";
    }

  ret = lw6sys_str_concat (dir, "/." PACKAGE_TARNAME "/" SCREENSHOT_FILE);

  return ret;
}

int
_lw6net_screenshot_init (_LW6NET_CONTEXT * context)
{
  int ret = 0;

  // we assume the screenshot struct is already zeroed
  context->screenshot.mutex = lw6sys_mutex_create ();
  if (context->screenshot.mutex)
    {
      ret =
	_lw6net_screenshot_refresh (context, &(context->screenshot), NULL,
				    NULL);
    }

  return ret;
}

void
_lw6net_screenshot_quit (_LW6NET_CONTEXT * context)
{
  lw6sys_mutex_destroy (context->screenshot.mutex);
}

static JSAMPLE *
create_screenshot (_LW6NET_CONTEXT * context,
		   _LW6NET_SCREENSHOT * screenshot,
		   LW6MAP_MAP * map,
		   LW6KER_GAME_STATE * game_state,
		   int *screenshot_width, int *screenshot_height)
{
  JSAMPLE *screenshot_buffer = NULL;
  int w, h;
  int x_dst, y_dst, x_src, y_src;
  int layer;
  int i_dst;
  LW6SYS_COLOR_8 color_8;
  LW6SYS_COLOR_F color_f;
  LW6KER_FIGHTER *fighter;
  float scale;
  float health;

  if (map && game_state && map->depth.shape.w > 0 && map->depth.shape.h > 0)
    {
      scale =
	sqrt (((float)
	       (context->const_data.screenshot_width *
		context->const_data.screenshot_height)) /
	      ((float) (map->depth.shape.w * map->depth.shape.h)));
      w = lw6sys_max (1, map->depth.shape.w * scale);
      h = lw6sys_max (1, map->depth.shape.h * scale);
    }
  else
    {
      w = context->const_data.screenshot_width;
      h = context->const_data.screenshot_height;
    }

  (*screenshot_width) = w;
  (*screenshot_height) = h;

  screenshot_buffer =
    (JSAMPLE *) LW6SYS_CALLOC (w * h * 3 * sizeof (JSAMPLE));

  if (map && game_state && map->depth.shape.w > 0 && map->depth.shape.h > 0)
    {
      for (y_dst = 0; y_dst < h; ++y_dst)
	{
	  for (x_dst = 0; x_dst < w; ++x_dst)
	    {
	      i_dst = (y_dst * w + x_dst) * 3;

	      x_src = (x_dst * map->texture.shape.w) / w;
	      y_src = (y_dst * map->texture.shape.h) / h;
	      color_8 = lw6map_texture_get (&(map->texture), x_src, y_src);
	      screenshot_buffer[i_dst] = color_8.r;
	      screenshot_buffer[i_dst + 1] = color_8.g;
	      screenshot_buffer[i_dst + 2] = color_8.b;

	      for (layer = game_state->map.nb_layers - 1; layer >= 0; layer--)
		{
		  x_src = (x_dst * game_state->map.shape.w) / w;
		  y_src = (y_dst * game_state->map.shape.h) / h;

		  fighter =
		    lw6ker_map_state_get_fighter_safe (&(game_state->map),
						       layer, x_src, y_src);
		  if (fighter)
		    {
		      lw6map_style_get_team_color (&(map->param.style),
						   &color_f,
						   fighter->team_id);
		      color_8 = lw6sys_color_f_to_8 (&color_f);
		      health =
			((float) fighter->health) / LW6OPT_MAX_FIGHTER_HEALTH;
		      screenshot_buffer[i_dst] = color_8.r * health;
		      screenshot_buffer[i_dst + 1] = color_8.g * health;
		      screenshot_buffer[i_dst + 2] = color_8.b * health;
		    }
		}
	    }
	}
    }
  else
    {
      // black JPEG
    }

  return screenshot_buffer;
}

static int
write_screenshot (_LW6NET_CONTEXT * context,
		  _LW6NET_SCREENSHOT * screenshot,
		  JSAMPLE * screenshot_buffer,
		  int screenshot_width, int screenshot_height, char *filename)
{
  int ret = 0;
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  FILE *outfile;
  JSAMPROW row_pointer[1];
  int row_stride;

  lw6sys_mutex_lock (screenshot->mutex);

  cinfo.err = jpeg_std_error (&jerr);
  jpeg_create_compress (&cinfo);

  if ((outfile = fopen (filename, "wb")) == NULL)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "net", _("can't open jpeg file \"%s\""),
		  filename);
    }
  else
    {
      jpeg_stdio_dest (&cinfo, outfile);

      cinfo.image_width = screenshot_width;
      cinfo.image_height = screenshot_height;
      cinfo.input_components = 3;
      cinfo.in_color_space = JCS_RGB;

      jpeg_set_defaults (&cinfo);
      jpeg_set_quality (&cinfo, context->const_data.screenshot_quality, TRUE);

      jpeg_start_compress (&cinfo, TRUE);

      row_stride = screenshot_width * 3;

      while (cinfo.next_scanline < cinfo.image_height)
	{
	  row_pointer[0] =
	    &screenshot_buffer[cinfo.next_scanline * row_stride];
	  (void) jpeg_write_scanlines (&cinfo, row_pointer, 1);
	}

      jpeg_finish_compress (&cinfo);

      ret = 1;

      fclose (outfile);
    }

  jpeg_destroy_compress (&cinfo);

  lw6sys_mutex_unlock (screenshot->mutex);

  return ret;
}

int
_lw6net_screenshot_refresh (_LW6NET_CONTEXT * context,
			    _LW6NET_SCREENSHOT * screenshot,
			    LW6MAP_MAP * map, LW6KER_GAME_STATE * game_state)
{
  int ret = 0;
  time_t now;
  JSAMPLE *screenshot_buffer;
  int screenshot_width;
  int screenshot_height;
  char *screenshot_filename;
  now = time (NULL);
  if (now >
      screenshot->last_refresh + context->const_data.screenshot_refresh_delay)
    {
      screenshot_buffer =
	create_screenshot (context, screenshot, map, game_state,
			   &screenshot_width, &screenshot_height);
      if (screenshot_buffer)
	{
	  screenshot_filename = lw6net_screenshot_get_default_jpeg ();
	  if (screenshot_filename)
	    {
	      if (write_screenshot
		  (context, screenshot, screenshot_buffer, screenshot_width,
		   screenshot_height, screenshot_filename))
		{
		  screenshot->last_refresh = now;
		  ret = 1;
		}
	      LW6SYS_FREE (screenshot_filename);
	    }
	  LW6SYS_FREE (screenshot_buffer);
	}
    }
  else
    {
      ret = 1;
    }

  return ret;
}
