/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>
#include <stdio.h>

#include "config.h"
#include "net.h"
#include "net-internal.h"

#define DOCUMENT_ROOT "net/htdocs"

static int
send_empty_line (_LW6NET_SERVER_HANDLE_DATA * handle_data)
{
  int ret = 0;

  ret =
    lw6net_send_line ((void *) handle_data->context, handle_data->sock, "");

  return ret;
}

static int
send_and_free_line (_LW6NET_SERVER_HANDLE_DATA * handle_data, char *line)
{
  int ret = 0;

  if (line)
    {
      ret =
	lw6net_send_line ((void *) handle_data->context, handle_data->sock,
			  line);
      LW6SYS_FREE (line);
    }

  return ret;
}

static int
send_headers (_LW6NET_SERVER_HANDLE_DATA * handle_data, int status_code,
	      char *content_type, int content_length,
	      _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret = 0;
  char *status_str = NULL;
  char *line;

  httpd_log_entry->status = status_code;
  httpd_log_entry->length = content_length;

  switch (status_code)
    {
    case 200:
      status_str = handle_data->context->const_data.http_status_200;
      break;
    case 404:
      status_str = handle_data->context->const_data.http_status_404;
      break;
    default:
      status_str = handle_data->context->const_data.http_status_500;
    }

  if (status_str)
    {
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.http_status,
			    status_code, status_str);
      send_and_free_line (handle_data, line);
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.
			    http_header_content_type, content_type);
      send_and_free_line (handle_data, line);
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.
			    http_header_content_length, content_length);
      send_and_free_line (handle_data, line);
      // insert Last-Modified here
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.
			    http_header_connection);
      send_and_free_line (handle_data, line);
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.
			    http_header_server, PACKAGE_TARNAME,
			    PACKAGE_VERSION);
      send_and_free_line (handle_data, line);
      line =
	lw6sys_new_sprintf (handle_data->context->const_data.
			    http_header_x_powered_by, PACKAGE_TARNAME);
      send_and_free_line (handle_data, line);

      ret = send_empty_line (handle_data);
    }

  return ret;
}

static int
send_binary (_LW6NET_SERVER_HANDLE_DATA * handle_data, int status_code,
	     char *content_type, int content_length, char *content,
	     _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret = 0;

  ret =
    send_headers (handle_data, status_code, content_type, content_length,
		  httpd_log_entry);

  ret = ret
    && lw6net_socket_send ((void *) handle_data->context, handle_data->sock,
			   content, content_length,
			   handle_data->context->const_data.http_delay, 1);

  return ret;
}

static int
send_text (_LW6NET_SERVER_HANDLE_DATA * handle_data, int status_code,
	   char *content_type, char *content,
	   _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret;

  ret =
    send_binary (handle_data, status_code, content_type, strlen (content),
		 content, httpd_log_entry);

  return ret;
}

int
_lw6net_http_response_send_binary (_LW6NET_SERVER_HANDLE_DATA * handle_data,
				   char *content_type, int content_length,
				   char *content,
				   _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret;

  ret =
    send_binary (handle_data, 200, content_type, content_length, content,
		 httpd_log_entry);

  return ret;
}

int
_lw6net_http_response_send_text (_LW6NET_SERVER_HANDLE_DATA * handle_data,
				 char *content_type, char *content,
				 _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret;

  ret = send_text (handle_data, 200, content_type, content, httpd_log_entry);

  return ret;
}

int
_lw6net_http_response_send_absolute_file (_LW6NET_SERVER_HANDLE_DATA *
					  handle_data, char *content_type,
					  char *filename,
					  void *file_mutex,
					  _LW6NET_HTTPD_LOG_ENTRY *
					  httpd_log_entry)
{
  int ret = 0;
  char *buffer = NULL;
  FILE *f = NULL;
  int size = 0;

  if (file_mutex)
    {
      lw6sys_mutex_lock (file_mutex);
    }
  f = fopen (filename, "r");
  if (f)
    {
      fseek (f, 0, SEEK_END);
      size = ftell (f);
      if (size > 0)
	{
	  buffer = LW6SYS_MALLOC (sizeof (char) * size);
	  if (buffer)
	    {
	      fseek (f, 0, SEEK_SET);
	      size = fread (buffer, sizeof (char), size, f);
	      if (size > 0)
		{
		  ret = 1;
		}
	    }
	}
    }
  if (file_mutex)
    {
      lw6sys_mutex_unlock (file_mutex);
    }

  if (ret)
    {
      ret = _lw6net_http_response_send_binary
	(handle_data, content_type, size, buffer, httpd_log_entry);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "net",
		  _("unable to open file \"%s\" for httpd server"), filename);
    }

  if (f)
    {
      fclose (f);
    }

  if (buffer)
    {
      LW6SYS_FREE (buffer);
    }

  return ret;
}

int
_lw6net_http_response_send_file (_LW6NET_SERVER_HANDLE_DATA *
				 handle_data, char *content_type,
				 char *request_uri,
				 _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret = 0;
  char *pos;
  char *relative_filename = NULL;
  char *absolute_filename = NULL;

  pos = strrchr (request_uri, '/');
  if (!pos)
    {
      pos = strrchr (request_uri, '\\');
    }

  if (pos)
    {
      relative_filename = lw6sys_str_concat (DOCUMENT_ROOT, request_uri);

      if (relative_filename)
	{
	  absolute_filename = lw6sys_find_data_file (relative_filename);
	  if (absolute_filename)
	    {
	      ret =
		_lw6net_http_response_send_absolute_file (handle_data,
							  content_type,
							  absolute_filename,
							  NULL,
							  httpd_log_entry);
	    }
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "net",
		  _("unable to serve file \"%s\""), request_uri);
    }

  if (relative_filename)
    {
      LW6SYS_FREE (relative_filename);
    }
  if (absolute_filename)
    {
      LW6SYS_FREE (absolute_filename);
    }

  return ret;
}

int
_lw6net_http_response_send_template (_LW6NET_SERVER_HANDLE_DATA *
				     handle_data,
				     char *request_uri, LW6SYS_ASSOC * values,
				     _LW6NET_HTTPD_LOG_ENTRY *
				     httpd_log_entry)
{
  int ret = 0;
  char *content = NULL;

  content = _lw6net_template_replace (handle_data, request_uri, values);
  if (content)
    {
      ret =
	_lw6net_http_response_send_text (handle_data,
					 handle_data->context->const_data.
					 content_type_html, content,
					 httpd_log_entry);
      LW6SYS_FREE (content);
    }

  return ret;
}

int
_lw6net_http_response_send_error (_LW6NET_SERVER_HANDLE_DATA * handle_data,
				  int status_code,
				  _LW6NET_HTTPD_LOG_ENTRY * httpd_log_entry)
{
  int ret = 0;

  switch (status_code)
    {
    case 404:
      ret = send_text (handle_data, 404,
		       handle_data->context->const_data.content_type_txt,
		       handle_data->context->const_data.http_status_404,
		       httpd_log_entry);
      break;
    default:
      ret = send_text (handle_data, 500,
		       handle_data->context->const_data.content_type_txt,
		       handle_data->context->const_data.http_status_500,
		       httpd_log_entry);
      break;
    }

  return ret;
}
