/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>

#include <libpng12/png.h>

#include "config.h"
#include "map.h"
#include "map-internal.h"

static int
read_image (LW6MAP_TEXTURE * texture, _LW6MAP_IMAGE_RGBA * image,
	    LW6MAP_ELEVATION * elevation)
{
  int ret = 0;

  texture->shape.w = image->shape.w;
  texture->shape.h = image->shape.h;
  texture->data =
    (LW6SYS_COLOR_8 *) LW6SYS_MALLOC (texture->shape.w *
				      texture->shape.h *
				      sizeof (LW6SYS_COLOR_8 *));

  if (texture->data)
    {
      int row, col;
      LW6SYS_COLOR_8 value;

      for (row = 0; row < texture->shape.h; ++row)
	{
	  for (col = 0; col < texture->shape.w; ++col)
	    {
	      value.r = image->data[row][col * 4];
	      value.g = image->data[row][col * 4 + 1];
	      value.b = image->data[row][col * 4 + 2];
	      value.a = image->data[row][col * 4 + 3];

	      lw6map_texture_set (texture, col, row, value);
	    }
	}

      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "map",
		  _("unable to allocate memory for texture"));
    }

  _lw6map_rgba_clear (image);

  if (!ret)
    {
      lw6map_texture_clear (texture);
    }

  return ret;
}

static int
read_alpha_image (LW6MAP_TEXTURE * texture, _LW6MAP_IMAGE_RGBA * image)
{
  int ret = 0;

  if (texture->shape.w == image->shape.w &&
      texture->shape.h == image->shape.h)
    {
      int row, col;
      LW6SYS_COLOR_8 value;
      LW6SYS_UINT32 alpha;

      for (row = 0; row < texture->shape.h; ++row)
	{
	  for (col = 0; col < texture->shape.w; ++col)
	    {
	      value = lw6map_texture_get (texture, col, row);
	      alpha = (((LW6SYS_UINT32) image->data[row][col * 4]) +
		       ((LW6SYS_UINT32) image->data[row][col * 4 + 1]) +
		       ((LW6SYS_UINT32) image->data[row][col * 4 + 2])) / 3;
	      value.a = alpha;
	      lw6map_texture_set (texture, col, row, value);
	    }
	}

      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "map",
		  _
		  ("texture size (%dx%d) and alpha layer size (%dx%d) do not match"),
		  texture->shape.w, texture->shape.h, image->shape.w,
		  image->shape.h);
    }

  _lw6map_rgba_clear (image);

  return ret;
}

static int
read_png (LW6MAP_TEXTURE * texture, char *texture_dot_png,
	  LW6MAP_ELEVATION * elevation)
{
  int ret = 0;
  _LW6MAP_IMAGE_RGBA image;

  memset (&image, 0, sizeof (_LW6MAP_IMAGE_RGBA));

  if (_lw6map_rgba_read_png (&image, texture_dot_png))
    {
      ret = read_image (texture, &image, elevation);
    }

  return ret;
}

static int
read_jpeg (LW6MAP_TEXTURE * texture, char *texture_dot_jpeg,
	   LW6MAP_ELEVATION * elevation)
{
  int ret = 0;
  _LW6MAP_IMAGE_RGBA image;

  memset (&image, 0, sizeof (_LW6MAP_IMAGE_RGBA));

  if (_lw6map_rgba_read_jpeg (&image, texture_dot_jpeg))
    {
      ret = read_image (texture, &image, elevation);
    }

  return ret;
}

static int
read_alpha_jpeg (LW6MAP_TEXTURE * texture, char *texture_alpha_dot_jpeg)
{
  int ret = 0;
  _LW6MAP_IMAGE_RGBA image;

  memset (&image, 0, sizeof (_LW6MAP_IMAGE_RGBA));

  if (_lw6map_rgba_read_jpeg (&image, texture_alpha_dot_jpeg))
    {
      ret = read_alpha_image (texture, &image);
    }

  return ret;
}

static int
default_texture (LW6MAP_TEXTURE * texture, LW6MAP_DEPTH * depth,
		 LW6MAP_ELEVATION * elevation, LW6SYS_COLOR_8 fg,
		 LW6SYS_COLOR_8 bg)
{
  int ret = 0;

  // Todo...

  return ret;
}

/*
 * Read the texture associated to a map. Pointer to texture must be valid,
 * it's modified in-place. The function will automatically figure out
 * if texture.png exists or if we must use foreground.png/background.png.
 * Depth and elevation are read-only.
 */
int
lw6map_texture_read (LW6MAP_TEXTURE * texture, char *dirname,
		     LW6MAP_DEPTH * depth, LW6MAP_ELEVATION * elevation)
{
  int ret = 0;

  lw6map_texture_clear (texture);

  if (!ret)
    {
      /*
       * First, we try to read texture.png
       */
      char *texture_dot_png;

      texture_dot_png = lw6sys_str_concat (dirname, LW6MAP_FILE_TEXTURE_PNG);
      if (texture_dot_png)
	{
	  if (lw6sys_file_exists (texture_dot_png))
	    {
	      ret = read_png (texture, texture_dot_png, elevation);
	    }
	  LW6SYS_FREE (texture_dot_png);
	}
    }

  if (!ret)
    {
      /*
       * Second, we try to read texture.jpeg
       */
      char *texture_dot_jpeg;

      texture_dot_jpeg =
	lw6sys_str_concat (dirname, LW6MAP_FILE_TEXTURE_JPEG);
      if (texture_dot_jpeg)
	{
	  if (lw6sys_file_exists (texture_dot_jpeg))
	    {
	      ret = read_jpeg (texture, texture_dot_jpeg, elevation);
	    }
	  LW6SYS_FREE (texture_dot_jpeg);
	}
    }

  if (!ret)
    {
      /*
       * Then, we try to generate a texture automatically
       * if this is the wish of the level designer
       */
      if (0)			// test existence of color defs in XML file here
	{
	  LW6SYS_COLOR_8 fg = { 0, 0, 0, 255 };	// could be loaded from XML
	  LW6SYS_COLOR_8 bg = { 255, 255, 255, 255 };	// could be loaded from XML

	  ret = default_texture (texture, depth, elevation, fg, bg);
	}
    }

  if (!ret)
    {
      /*
       * Finally, we read map.png, which should always be here.
       */
      char *map_dot_png;

      map_dot_png = lw6sys_str_concat (dirname, LW6MAP_FILE_MAP_PNG);
      if (map_dot_png)
	{
	  if (lw6sys_file_exists (map_dot_png))
	    {
	      ret = read_png (texture, map_dot_png, elevation);
	    }
	  LW6SYS_FREE (map_dot_png);
	}
    }

  if (ret)
    {
      /*
       * If texture is successfully loaded, apply separate
       * JPEG alpha channel.
       */
      char *texture_alpha_dot_jpeg;

      texture_alpha_dot_jpeg =
	lw6sys_str_concat (dirname, LW6MAP_FILE_TEXTURE_ALPHA_JPEG);
      if (texture_alpha_dot_jpeg)
	{
	  if (lw6sys_file_exists (texture_alpha_dot_jpeg))
	    {
	      ret = read_alpha_jpeg (texture, texture_alpha_dot_jpeg);
	    }
	  LW6SYS_FREE (texture_alpha_dot_jpeg);
	}
    }

  return ret;
}

/*
 * Clears a texture.
 */
void
lw6map_texture_clear (LW6MAP_TEXTURE * texture)
{
  if (texture->data)
    {
      LW6SYS_FREE (texture->data);
    }

  memset (texture, 0, sizeof (LW6MAP_TEXTURE));
}
