/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "liquidwar6.h"

/*
 * Helper function, creates a 0 terminated string
 * from a Guile string. A very common task.
 * Returned pointer must be freed.
 */
static char *
to_0str (SCM string)
{
  char *c_string;
  int length;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, "");

  /*
   * See the comment in sys/sys-str.c to see why we use
   * 2 trailing '\0' at the end of the string.
   */
  length = scm_c_string_length (string);
  c_string = (char *) LW6SYS_MALLOC ((length + 2) * sizeof (char));

  if (c_string)
    {
      memcpy ((void *) c_string, (void *) scm_i_string_chars (string),
	      length * sizeof (char));
      c_string[length] = c_string[length + 1] = '\0';
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "",
		  _
		  ("unable to convert a guile SCM string to a standard C \"'\\0' terminated\" string"));
    }

  return c_string;
}

/*
 * For a GNU gettext-like behavior of scheme code.
 */
static SCM
scm_gettext (SCM string)
{
  char *c_string;
  SCM ret;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, "c-gettext");

  c_string = to_0str (string);
  if (c_string)
    {
      ret = scm_makfrom0str (gettext (c_string));
      LW6SYS_FREE (c_string);
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In liquidwar6sys
 */

/*
 * In log.c
 */

static SCM
scm_lw6sys_log (SCM level, SCM message)
{
  int c_level;
  char *c_message;

  SCM_ASSERT (scm_is_integer (level), level, SCM_ARG1, "c-lw6sys-log");
  SCM_ASSERT (scm_is_string (message), message, SCM_ARG2, "c-lw6sys-log");

  c_level = scm_to_int (level);
  c_message = to_0str (message);
  if (c_message)
    {
      lw6sys_log (c_level, __FILE__, __LINE__, "script", "%s", c_message);
      LW6SYS_FREE (c_message);
    }

  return SCM_UNDEFINED;
}

/*
 * In time.c
 */
static SCM
scm_lw6sys_timestamp ()
{
  SCM ret;

  ret = scm_long_long2num (lw6sys_timestamp ());

  return ret;
}

/*
 * In liquidwar6cfg
 */

/*
 * In commandline.c
 */
static SCM
scm_lw6cfg_parse_command_line ()
{
  if (lw6_global.cfg_context != NULL)
    {
      lw6cfg_parse_command_line (lw6_global.cfg_context, lw6_global.argc,
				 lw6_global.argv);
    };

  return SCM_UNDEFINED;
}

/*
 * In dir.c
 */
static SCM
scm_lw6cfg_get_default_system_config_file ()
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      ret =
	scm_makfrom0str (lw6cfg_get_default_system_config_file
			 (lw6_global.cfg_context));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6cfg_get_default_user_config_file ()
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      ret =
	scm_makfrom0str (lw6cfg_get_default_user_config_file
			 (lw6_global.cfg_context));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In load.c
 */
static SCM
scm_lw6cfg_load (SCM filename)
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename),
		  filename, SCM_ARG1, "c-lw6cfg-load");

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_load (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In option.c
 */
static SCM
scm_lw6cfg_option_exists (SCM key)
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1,
		  "c-lw6cfg-option-exists");

      c_key = to_0str (key);
      if (c_key)
	{
	  if (lw6cfg_option_exists (lw6_global.cfg_context, c_key))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_key);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6cfg_get_option (SCM key)
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, "c-lw6cfg-get-option");

      c_key = to_0str (key);
      if (c_key)
	{
	  char *value = NULL;

	  value = lw6cfg_get_option (lw6_global.cfg_context, c_key);
	  if (value)
	    {
	      ret =
		scm_makfrom0str (lw6cfg_get_option
				 (lw6_global.cfg_context, c_key));
	    }
	  else
	    {
	      ret = SCM_UNDEFINED;
	    }

	  LW6SYS_FREE (c_key);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6cfg_set_option (SCM key, SCM value)
{
  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;
      char *c_value;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, "c-lw6cfg-set-option");

      SCM_ASSERT (scm_is_string (value),
		  value, SCM_ARG2, "c-lw6cfg-set-option");

      c_key = to_0str (key);
      if (c_key)
	{
	  c_value = to_0str (value);
	  if (c_value)
	    {
	      lw6cfg_set_option (lw6_global.cfg_context, c_key, c_value);

	      LW6SYS_FREE (c_value);
	    }
	  LW6SYS_FREE (c_key);
	}
    }

  return SCM_UNDEFINED;
}

/*
 * In save.c
 */
static SCM
scm_lw6cfg_save (SCM filename)
{
  SCM ret;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename),
		  filename, SCM_ARG1, "c-lw6cfg-save");

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_save (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In setup.c
 */
static SCM
scm_lw6cfg_init ()
{
  SCM ret;

  lw6_global.cfg_context = lw6cfg_init ();
  lw6_global.cfg_initialized = lw6_global.cfg_context ? 1 : 0;
  ret = lw6_global.cfg_context ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6cfg_quit ()
{
  if (lw6_global.cfg_context != NULL)
    {
      lw6cfg_quit (lw6_global.cfg_context);
    }

  lw6_global.cfg_context = NULL;
  lw6_global.cfg_initialized = 0;

  return SCM_UNDEFINED;
}


/*
 * In liquidwar6opt
 */

/*
 * In static.c
 */
static SCM
scm_lw6opt_static_get_options_list ()
{
  SCM ret = SCM_EOL;
  int i = 0;

  while (LW6OPT_STATIC_OPTIONS_LIST[i])
    {
      ret = scm_cons (scm_makfrom0str (LW6OPT_STATIC_OPTIONS_LIST[i]), ret);
      ++i;
    }
  ret = scm_reverse (ret);

  return ret;
}

static SCM
scm_lw6opt_static_get_int (SCM game_struct, SCM key)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STRUCT *c_game_struct;
      char *c_key;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG1,
		  "c-lw6opt-static-get-int");
      SCM_ASSERT (scm_is_string (key), key, SCM_ARG2,
		  "c-lw6opt-static-get-int");

      c_game_struct = lw6_scm_to_game_struct (game_struct);
      if (c_game_struct)
	{
	  c_key = to_0str (key);
	  if (c_key)
	    {
	      ret =
		scm_int2num (lw6opt_static_get_int
			     (&(c_game_struct->options), c_key));
	      LW6SYS_FREE (c_key);
	    }
	}
    }

  return ret;
}

/*
 * In liquidwar6gfx
 */

/*
 * In backend.c
 */
static SCM
scm_lw6gfx_get_backends ()
{
  SCM ret = SCM_UNDEFINED;
  LW6SYS_ASSOC *backends;
  LW6SYS_LIST *keys;
  LW6SYS_LIST *key;
  char *module_id;
  char *module_name;

  backends = lw6gfx_get_backends ();
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  return ret;
}

static SCM
scm_lw6gfx_create_backend (SCM backend_name)
{
  SCM ret;

  if (lw6_global.gfx_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, "c-lw6gfx-create-backend");

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.gfx_backend = lw6gfx_create_backend (c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.gfx_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_destroy_backend ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      if (lw6_global.gfx_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "",
		      _
		      ("destroying gfx backend, but backend has not been previously cleaned up"));
	  lw6gfx_quit (lw6_global.gfx_backend);
	}

      lw6gfx_destroy_backend (lw6_global.gfx_backend);
    }
  lw6_global.gfx_backend = NULL;
  lw6_global.gfx_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In background.c
 */
static SCM
scm_lw6gfx_display_background (SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;

      SCM_ASSERT (scm_is_string (skin),
		  skin, SCM_ARG1, "c-lw6gfx-display-background");

      c_skin = to_0str (skin);
      if (c_skin)
	{
	  lw6gfx_display_background (lw6_global.gfx_backend, c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

/*
 * In event.c
 */
static SCM
scm_lw6gfx_poll_keypress ()
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6GFX_KEYPRESS keypress;

      if (lw6gfx_poll_keypress (lw6_global.gfx_backend, &keypress))
	{
	  ret =
	    scm_list_3 (scm_cons
			(scm_makfrom0str ("sym"), scm_int2num (keypress.sym)),
			scm_cons (scm_makfrom0str ("unicode"),
				  scm_int2num (keypress.unicode)),
			scm_cons (scm_makfrom0str ("label"),
				  scm_makfrom0str (keypress.label)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_esc (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-esc");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_esc (lw6_global.gfx_backend,
			   c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_up (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-up");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_up (lw6_global.gfx_backend,
			  c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_down (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-down");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_down (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_left (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-left");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_left (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_right (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-right");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_right (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_enter (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-enter");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_enter (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_del (SCM keysym)
{
  SCM ret;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-del");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_del (lw6_global.gfx_backend,
			   c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_backspace (SCM keysym)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-backspace");
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_is_key_backspace (lw6_global.gfx_backend,
				 c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_help (SCM keysym)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-help");
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_is_key_help (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_quit (SCM keysym)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-quit");
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_is_key_quit (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_quit ()
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_quit (lw6_global.gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_send_quit ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_send_quit (lw6_global.gfx_backend);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_get_key_state (SCM keysym)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-get-key-state");
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_get_key_state (lw6_global.gfx_backend,
			      c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_move ()
{
  SCM ret = SCM_UNDEFINED;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;

      if (lw6gfx_poll_mouse_move (lw6_global.gfx_backend, &c_x, &c_y))
	{
	  ret = scm_list_2 (scm_cons
			    (scm_makfrom0str ("x"), scm_int2num (c_x)),
			    scm_cons (scm_makfrom0str ("y"),
				      scm_int2num (c_y)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_button (SCM button)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_button;
      SCM_ASSERT (scm_is_integer (button), button, SCM_ARG1,
		  "c-lw6gfx-poll-mouse-button");
      c_button = scm_to_int (button);
      ret =
	lw6gfx_poll_mouse_button (lw6_global.gfx_backend,
				  c_button) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_get_mouse_state ()
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;
      int c_button_left;
      int c_button_right;
      lw6gfx_get_mouse_state (lw6_global.gfx_backend,
			      &c_x, &c_y, &c_button_left, &c_button_right);
      ret =
	scm_list_4 (scm_cons
		    (scm_makfrom0str ("x"),
		     scm_int2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"),
			      scm_int2num (c_y)),
		    scm_cons (scm_makfrom0str
			      ("button-left"),
			      c_button_left ? SCM_BOOL_T :
			      SCM_BOOL_F),
		    scm_cons (scm_makfrom0str
			      ("button-right"),
			      c_button_right ? SCM_BOOL_T : SCM_BOOL_F));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In hud.c
 */
static SCM
scm_lw6gfx_display_hud (SCM game_state, SCM step, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int c_step;
      char *c_skin;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6gfx-display-hud");
      SCM_ASSERT (scm_is_integer (step), step, SCM_ARG2,
		  "c-lw6gfx-display-hud");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG3,
		  "c-lw6gfx-display-hud");

      c_step = scm_to_int (step);
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6KER_GAME_STATE *c_game_state;
	  c_game_state = lw6_scm_to_game_state (game_state);
	  lw6gfx_display_hud (lw6_global.gfx_backend, c_game_state, c_step,
			      c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_hud_preview (SCM step, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int c_step;
      char *c_skin;

      SCM_ASSERT (scm_is_integer (step), step, SCM_ARG1,
		  "c-lw6gfx-display-hud-preview");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG2,
		  "c-lw6gfx-display-hud-preview");

      c_step = scm_to_int (step);
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  lw6gfx_display_hud_preview (lw6_global.gfx_backend, c_step, c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

/*
 * In info.c
 */
static SCM
scm_lw6gfx_display_sysinfo (SCM fps, SCM rps, SCM bytes_in, SCM bytes_out)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int c_fps;
      int c_rps;
      int c_bytes_in;
      int c_bytes_out;
      SCM_ASSERT (scm_is_integer (fps), fps, SCM_ARG1,
		  "c-lw6sys-display-sysinfo");
      SCM_ASSERT (scm_is_integer (rps), rps, SCM_ARG2,
		  "c-lw6sys-display-sysinfo");
      SCM_ASSERT (scm_is_integer (bytes_in), bytes_in, SCM_ARG3,
		  "c-lw6sys-display-sysinfo");
      SCM_ASSERT (scm_is_integer (bytes_out), bytes_out, SCM_ARG4,
		  "c-lw6sys-display-sysinfo");
      c_fps = scm_to_int (fps);
      c_rps = scm_to_int (rps);
      c_bytes_in = scm_to_int (bytes_in);
      c_bytes_out = scm_to_int (bytes_out);
      lw6gfx_display_sysinfo (lw6_global.gfx_backend,
			      c_fps, c_rps, c_bytes_in, c_bytes_out);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_log (SCM log)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int log_size;
      char **c_log;
      SCM_ASSERT (SCM_CONSP (log), log, SCM_ARG1, "c-lw6sys-display-log");
      log_size = scm_to_int (scm_length (log));
      c_log = LW6SYS_MALLOC ((log_size + 1) * sizeof (char *));
      if (c_log)
	{
	  int i;
	  for (i = 0; i < log_size; ++i)
	    {
	      SCM str;
	      str = scm_list_ref (log, scm_int2num (i));
	      c_log[i] = to_0str (str);
	    }
	  c_log[log_size] = NULL;
	  lw6gfx_display_log (lw6_global.gfx_backend, c_log);
	  for (i = 0; i < log_size; ++i)
	    {
	      if (c_log[i])
		{
		  LW6SYS_FREE (c_log[i]);
		}
	    }

	  LW6SYS_FREE (c_log);
	}
    }

  return SCM_UNDEFINED;
}


static SCM
scm_lw6gfx_display_time (SCM seconds)
{
  if (lw6_global.gfx_backend != NULL)
    {
      long long c_seconds;
      SCM_ASSERT (SCM_NUMBERP (seconds), seconds,
		  SCM_ARG1, "c-lw6sys-display-time");
      c_seconds = scm_num2long_long (seconds, 0, "c-lw6sys-display-time");
      lw6gfx_display_time (lw6_global.gfx_backend, c_seconds);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_version ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_display_version (lw6_global.gfx_backend);
    }

  return SCM_UNDEFINED;
}

/*
 * In map.c
 */
static SCM
scm_lw6gfx_display_map_preview (SCM map, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map), map,
		  SCM_ARG1, "c-lw6gfx-display-map-preview");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG2,
		  "c-lw6gfx-display-map-preview");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6MAP_MAP *c_map;
	  c_map = lw6_scm_to_map (map);
	  lw6gfx_display_map_preview (lw6_global.gfx_backend, c_map, c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_map_zones (SCM map, SCM game_struct, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map,
		   map), map, SCM_ARG1, "c-lw6gfx-display-map-zones");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG2,
		  "c-lw6gfx-display-map-zones");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG3,
		  "c-lw6gfx-display-map-zones");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6MAP_MAP *c_map;
	  LW6KER_GAME_STRUCT *c_game_struct;

	  c_map = lw6_scm_to_map (map);
	  c_game_struct = lw6_scm_to_game_struct (game_struct);

	  lw6gfx_display_map_zones (lw6_global.gfx_backend,
				    c_map, &(c_game_struct->map), c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_map_gradient (SCM map, SCM game_state, SCM team_id,
				 SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map,
		   map), map, SCM_ARG1, "c-lw6gfx-display-map-gradient");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2,
		  "c-lw6gfx-display-map-gradient");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG3,
		  "c-lw6gfx-display-map-gradient");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG4,
		  "c-lw6gfx-display-map-gradient");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6MAP_MAP *c_map;
	  LW6KER_GAME_STATE *c_game_state;
	  int c_team_id;

	  c_map = lw6_scm_to_map (map);
	  c_game_state = lw6_scm_to_game_state (game_state);
	  c_team_id = scm_to_int (team_id);

	  lw6gfx_display_map_gradient (lw6_global.
				       gfx_backend,
				       c_map,
				       &(c_game_state->map),
				       c_team_id, c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_game (SCM map, SCM game_state, SCM game_look, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map),
		  map, SCM_ARG1, "c-lw6gfx-display-game");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2,
		  "c-lw6gfx-display-game");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_look,
		   game_look), game_look, SCM_ARG3, "c-lw6gfx-display-game");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG4,
		  "c-lw6gfx-display-game");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6MAP_MAP *c_map;
	  LW6KER_GAME_STATE *c_game_state;
	  LW6GFX_GAME_LOOK *c_game_look;
	  c_map = lw6_scm_to_map (map);
	  c_game_state = lw6_scm_to_game_state (game_state);
	  c_game_look = lw6_scm_to_game_look (game_look);
	  lw6gfx_display_game (lw6_global.gfx_backend, c_map,
			       c_game_state, c_game_look, c_skin);
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_pick_map_coord (SCM map, SCM game_state, SCM screen_x,
			   SCM screen_y, SCM skin)
{
  SCM ret = SCM_UNDEFINED;

  if (lw6_global.gfx_backend != NULL)
    {
      char *c_skin;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map),
		  map, SCM_ARG1, "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (scm_is_integer (screen_x), screen_x, SCM_ARG3,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (scm_is_integer (screen_y), screen_y, SCM_ARG4,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG5,
		  "c-lw6gfx-pick-map-coord");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  LW6MAP_MAP *c_map;
	  LW6KER_GAME_STATE *c_game_state;
	  int c_screen_x;
	  int c_screen_y;
	  int c_map_x;
	  int c_map_y;

	  c_map = lw6_scm_to_map (map);
	  c_game_state = lw6_scm_to_game_state (game_state);
	  c_screen_x = scm_to_int (screen_x);
	  c_screen_y = scm_to_int (screen_y);

	  lw6gfx_pick_map_coord (lw6_global.gfx_backend, &c_map_x, &c_map_y,
				 c_map, c_game_state, c_screen_x, c_screen_y,
				 c_skin);

	  ret = scm_list_2 (scm_cons
			    (scm_makfrom0str ("x"), scm_int2num (c_map_x)),
			    scm_cons (scm_makfrom0str ("y"),
				      scm_int2num (c_map_y)));

	  LW6SYS_FREE (c_skin);
	}
    }

  return ret;
}

/*
 * In menu.c
 */
static SCM
scm_lw6gfx_display_menu (SCM menu, SCM skin)
{
  if (lw6_global.gfx_backend != NULL)
    {
      LW6GFX_MENU *c_menu;
      char *c_skin;
      SCM items;
      SCM_ASSERT (SCM_CONSP (menu)
		  || menu == SCM_EOL, menu, SCM_ARG1,
		  "c-lw6gfx-display-menu");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG2,
		  "c-lw6gfx-display-menu");
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  char *c_id;
	  char *c_label;
	  LW6GFX_MENU_TYPE c_type;
	  int c_selected_item;
	  int c_nb_items;

	  c_id = to_0str (scm_assoc_ref (menu, scm_makfrom0str ("id")));
	  c_label = to_0str (scm_assoc_ref (menu, scm_makfrom0str ("label")));
	  c_type =
	    scm_to_int (scm_assoc_ref (menu, scm_makfrom0str ("type")));
	  c_selected_item =
	    scm_to_int (scm_assoc_ref
			(menu, scm_makfrom0str ("selected-item")));
	  items = scm_assoc_ref (menu, scm_makfrom0str ("items"));
	  c_nb_items = scm_to_int (scm_length (items));

	  c_menu =
	    lw6gfx_menu_new (c_id, c_label, c_type, c_selected_item,
			     c_nb_items);
	  LW6SYS_FREE (c_id);
	  LW6SYS_FREE (c_label);
	  if (c_menu)
	    {
	      int i;
	      for (i = 0; i < c_nb_items; ++i)
		{
		  LW6GFX_MENUITEM *c_menuitem;
		  SCM item;
		  char *c_label;
		  LW6GFX_MENUITEM_TYPE c_type;
		  int c_enabled;
		  int c_value;
		  int c_selected_char;

		  item = scm_list_ref (items, scm_int2num (i));
		  c_label =
		    to_0str (scm_assoc_ref (item, scm_makfrom0str ("label")));
		  c_type =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("type")));
		  c_enabled =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("enabled")));
		  c_value =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("value")));
		  c_selected_char =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("selected-char")));
		  c_menuitem =
		    lw6gfx_menuitem_new (c_label, c_type, c_enabled, c_value,
					 c_selected_char);
		  LW6SYS_FREE (c_label);
		  if (c_menuitem)
		    {
		      lw6gfx_menu_set_item (c_menu, i, c_menuitem);
		    }
		}

	      lw6gfx_display_menu (lw6_global.gfx_backend, c_menu, c_skin);
	      lw6gfx_menu_free (c_menu);
	    }
	  LW6SYS_FREE (c_skin);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_pick_menuitem (SCM menu, SCM screen_x, SCM screen_y, SCM skin)
{
  SCM ret = SCM_UNDEFINED;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6GFX_MENU *c_menu;
      int c_screen_x;
      int c_screen_y;
      char *c_skin;
      SCM items;
      SCM_ASSERT (SCM_CONSP (menu), menu, SCM_ARG1,
		  "c-lw6gfx-display-menuitem");
      SCM_ASSERT (scm_is_integer (screen_x), screen_x, SCM_ARG2,
		  "c-lw6gfx-pick-menuitem");
      SCM_ASSERT (scm_is_integer (screen_y), screen_y, SCM_ARG3,
		  "c-lw6gfx-pick-menuitem");
      SCM_ASSERT (scm_is_string (skin), skin, SCM_ARG4,
		  "c-lw6gfx-pick-menuitem");
      c_screen_x = scm_to_int (screen_x);
      c_screen_y = scm_to_int (screen_y);
      c_skin = to_0str (skin);
      if (c_skin)
	{
	  char *c_id;
	  char *c_label;
	  LW6GFX_MENU_TYPE c_type;
	  int c_selected_item;
	  int c_nb_items;

	  c_id = to_0str (scm_assoc_ref (menu, scm_makfrom0str ("id")));
	  c_label = to_0str (scm_assoc_ref (menu, scm_makfrom0str ("label")));
	  c_type =
	    scm_to_int (scm_assoc_ref (menu, scm_makfrom0str ("type")));
	  c_selected_item =
	    scm_to_int (scm_assoc_ref
			(menu, scm_makfrom0str ("selected-item")));
	  items = scm_assoc_ref (menu, scm_makfrom0str ("items"));
	  c_nb_items = scm_to_int (scm_length (items));

	  c_menu =
	    lw6gfx_menu_new (c_id, c_label, c_type, c_selected_item,
			     c_nb_items);
	  LW6SYS_FREE (c_id);
	  LW6SYS_FREE (c_label);
	  if (c_menu)
	    {
	      int i;
	      for (i = 0; i < c_nb_items; ++i)
		{
		  LW6GFX_MENUITEM *c_menuitem;
		  SCM item;
		  char *c_label;
		  LW6GFX_MENUITEM_TYPE c_type;
		  int c_enabled;
		  int c_value;
		  int c_selected_char;

		  item = scm_list_ref (items, scm_int2num (i));
		  c_label =
		    to_0str (scm_assoc_ref (item, scm_makfrom0str ("label")));
		  c_type =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("type")));
		  c_enabled =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("enabled")));
		  c_value =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("value")));
		  c_selected_char =
		    scm_to_int (scm_assoc_ref
				(item, scm_makfrom0str ("selected-char")));
		  c_menuitem =
		    lw6gfx_menuitem_new (c_label, c_type, c_enabled, c_value,
					 c_selected_char);
		  LW6SYS_FREE (c_label);
		  if (c_menuitem)
		    {
		      lw6gfx_menu_set_item (c_menu, i, c_menuitem);
		    }
		}

	      ret =
		scm_int2num (lw6gfx_pick_menuitem
			     (lw6_global.gfx_backend, c_menu, c_screen_x,
			      c_screen_y, c_skin));
	      lw6gfx_menu_free (c_menu);
	    }
	  LW6SYS_FREE (c_skin);
	}
    }

  return ret;
}

/*
 * In render.c
 */
static SCM
scm_lw6gfx_prepare_buffer ()
{
  lw6gfx_prepare_buffer (lw6_global.gfx_backend);
  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_swap_buffers ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_swap_buffers (lw6_global.gfx_backend);
    }

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
scm_lw6gfx_init (SCM width, SCM height, SCM fullscreen, SCM ticks)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      int c_ticks;
      SCM_ASSERT (scm_is_integer (width), width, SCM_ARG1, "c-lw6gfx-init");
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2, "c-lw6gfx-init");
      SCM_ASSERT (SCM_BOOLP (fullscreen), fullscreen, SCM_ARG3,
		  "c-lw6gfx-init");
      SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG4, "c-lw6gfx-init");
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      c_ticks = scm_to_int (ticks);
      if (lw6gfx_init (lw6_global.gfx_backend, c_width,
		       c_height, c_fullscreen, lw6_resize_callback, c_ticks))
	{
	  lw6_global.gfx_initialized = 1;
	  ret = SCM_BOOL_T;
	}
    }

  return ret;
}

static SCM
scm_lw6gfx_set_video_mode (SCM width, SCM height, SCM fullscreen)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      SCM_ASSERT (scm_is_integer (width),
		  width, SCM_ARG1, "c-lw6gfx-set-video-mode");
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2,
		  "c-lw6gfx-set-video-mode");
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      ret =
	lw6gfx_set_video_mode (lw6_global.gfx_backend,
			       c_width, c_height,
			       c_fullscreen,
			       lw6_resize_callback) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_quit ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_quit (lw6_global.gfx_backend);
    }
  lw6_global.gfx_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In timer.c
 */
static SCM
scm_lw6gfx_delay (SCM ms)
{
  int c_ms;
  if (lw6_global.gfx_backend != NULL)
    {
      SCM_ASSERT (scm_is_integer (ms), ms, SCM_ARG1, "c-lw6gfx-delay");
      c_ms = scm_to_int (ms);
      lw6gfx_delay (lw6_global.gfx_backend, c_ms);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_get_ticks ()
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      ret = scm_int2num (lw6gfx_get_ticks (lw6_global.gfx_backend));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6gfx_default_game_look ()
{
  LW6GFX_GAME_LOOK *c_game_look;
  SCM ret;
  c_game_look = lw6gfx_game_look_new ();
  if (c_game_look)
    {
      ret = lw6_make_scm_game_look (c_game_look);
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In liquidwar6map
 */


/*
 * In dir.c
 */
static SCM
scm_lw6map_get_system_dir ()
{
  SCM ret;
  char *c_system_dir;
  c_system_dir = lw6map_get_system_dir ();
  if (c_system_dir)
    {
      ret = scm_makfrom0str (c_system_dir);
      // LW6SYS_FREE(c_system_dir); (static pointer)
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6map_get_maps (SCM dir)
{
  LW6SYS_LIST *c_maps;
  LW6MAP_ENTRY *c_entry;
  char *c_dir;
  SCM ret;
  SCM_ASSERT (scm_is_string (dir), dir, SCM_ARG1, "c-lw6map-get-system-maps");
  ret = SCM_LIST0;
  c_dir = to_0str (dir);
  if (c_dir)
    {
      c_maps = lw6map_get_maps (c_dir);
      if (c_maps)
	{
	  while (!lw6sys_list_is_empty (c_maps))
	    {
	      c_entry = (LW6MAP_ENTRY *) lw6sys_list_pop (&c_maps);
	      if (c_entry)
		{
		  ret = scm_cons (scm_list_3
				  (scm_cons
				   (scm_makfrom0str ("title"),
				    scm_makfrom0str (c_entry->
						     title)),
				   scm_cons (scm_makfrom0str
					     ("path"),
					     scm_makfrom0str
					     (c_entry->path)),
				   scm_cons (scm_makfrom0str
					     ("subdir"),
					     c_entry->
					     subdir ? SCM_BOOL_T :
					     SCM_BOOL_F)), ret);
		  lw6map_free_entry (c_entry);
		}
	    }
	  lw6sys_list_free (c_maps);
	}
      LW6SYS_FREE (c_dir);
    }

  ret = scm_reverse (ret);
  return ret;
}

/*
 * In read.c
 */
static SCM
scm_lw6map_read (SCM dirname)
{
  char *c_dirname;
  SCM ret;

  SCM_ASSERT (scm_is_string (dirname), dirname, SCM_ARG1, "c-lw6map-read");

  c_dirname = to_0str (dirname);
  if (c_dirname)
    {
      LW6MAP_PARAM param;
      LW6MAP_MAP *c_map;

      // todo : get options from SCM obj
      memset (&param, 0, sizeof (LW6MAP_PARAM));
      lw6map_param_defaults (&param);
      c_map = lw6map_read (c_dirname, &param);
      if (c_map)
	{
	  ret = lw6_make_scm_map (c_map);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
      lw6map_param_clear (&param);
      LW6SYS_FREE (c_dirname);
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6map_get_background_style (SCM map)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6MAP_MAP *c_map;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map), map,
		  SCM_ARG1, "c-lw6gfx-get-background-style");
      c_map = lw6_scm_to_map (map);
      if (c_map->param.style.background_style)
	{
	  ret = scm_makfrom0str (c_map->param.style.background_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_hud_style (SCM map)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6MAP_MAP *c_map;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map), map,
		  SCM_ARG1, "c-lw6gfx-get-hud-style");
      c_map = lw6_scm_to_map (map);
      if (c_map->param.style.hud_style)
	{
	  ret = scm_makfrom0str (c_map->param.style.hud_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_menu_style (SCM map)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6MAP_MAP *c_map;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map), map,
		  SCM_ARG1, "c-lw6gfx-get-menu-style");
      c_map = lw6_scm_to_map (map);
      if (c_map->param.style.menu_style)
	{
	  ret = scm_makfrom0str (c_map->param.style.menu_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_view_style (SCM map)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6MAP_MAP *c_map;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, map), map,
		  SCM_ARG1, "c-lw6gfx-get-view-style");
      c_map = lw6_scm_to_map (map);
      if (c_map->param.style.view_style)
	{
	  ret = scm_makfrom0str (c_map->param.style.view_style);
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In liquidwar6ker
 */
static SCM
scm_lw6ker_build_game_struct (SCM map)
{
  LW6MAP_MAP *c_map;
  SCM ret;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.map, map), map,
	      SCM_ARG1, "c-lw6ker-build-game-struct");
  c_map = lw6_scm_to_map (map);
  if (c_map)
    {
      LW6KER_GAME_STRUCT *c_game_struct;

      c_game_struct = lw6ker_game_struct_new (c_map);
      if (c_game_struct)
	{
	  ret = lw6_make_scm_game_struct (c_game_struct);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_build_game_state (SCM game_struct)
{
  LW6KER_GAME_STRUCT *c_game_struct;
  SCM ret;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1,
	      "c-lw6ker-build-game-state");
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      LW6KER_GAME_STATE *c_game_state;
      c_game_state = lw6ker_game_state_new (c_game_struct);
      if (c_game_state)
	{
	  ret = lw6_make_scm_game_state (c_game_state);
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_copy_game_state (SCM dst, SCM src)
{
  LW6KER_GAME_STATE *c_dst;
  LW6KER_GAME_STATE *c_src;
  SCM ret;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       dst), dst, SCM_ARG1, "c-lw6ker-copy-game-state");
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       src), src, SCM_ARG2, "c-lw6ker-copy-game-state");
  c_dst = lw6_scm_to_game_state (dst);
  c_src = lw6_scm_to_game_state (src);
  if (c_dst && c_src)
    {
      ret = lw6ker_game_state_copy (c_dst, c_src) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_game_struct_checksum (SCM game_struct)
{
  LW6KER_GAME_STRUCT *c_game_struct;
  SCM ret;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1,
	      "c-lw6ker-game-struct-checksum");
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      ret = scm_int2num (lw6ker_game_struct_checksum (c_game_struct));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_game_state_checksum (SCM game_state)
{
  LW6KER_GAME_STATE *c_game_state;
  SCM ret;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1,
	      "c-lw6ker-game-state-checksum");
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_checksum (c_game_state));
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_get_cursors (SCM game_state, SCM team_id)
{
  LW6KER_GAME_STATE *c_game_state;
  int c_team_id;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-get-cursors");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-get-cursors");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      int i;
      LW6KER_CURSOR_ARRAY *c_cursor_array =
	&(c_game_state->map.teams[c_team_id].cursor_array);
      ret = SCM_EOL;

      for (i = 0; i < c_cursor_array->nb_cursors; ++i)
	{
	  ret =
	    scm_cons (scm_list_3 (scm_cons (scm_makfrom0str ("x"),
					    scm_int2num (c_cursor_array->
							 cursors[i].pos.x)),
				  scm_cons (scm_makfrom0str ("y"),
					    scm_int2num (c_cursor_array->
							 cursors[i].pos.y)),
				  scm_cons (scm_makfrom0str ("pot-offset"),
					    scm_int2num (c_cursor_array->
							 cursors[i].
							 pot_offset))), ret);
	}
      ret = scm_reverse (ret);
    }

  return ret;
}

static SCM
scm_lw6ker_set_cursors (SCM game_state, SCM team_id, SCM cursor_array)
{
  LW6KER_GAME_STATE *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-set-cursors");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-set-cursors");
  SCM_ASSERT (SCM_CONSP (cursor_array)
	      || cursor_array == SCM_EOL, cursor_array, SCM_ARG3,
	      "c-lw6ker-set-cursors");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      int i = 0;
      int n;
      LW6KER_CURSOR_ARRAY *c_cursor_array =
	&(c_game_state->map.teams[c_team_id].cursor_array);
      SCM cursor;

      n = scm_to_int (scm_length (cursor_array));
      for (i = 0; i < n; ++i)
	{
	  cursor = scm_list_ref (cursor_array, scm_int2num (i));
	  lw6ker_cursor_array_set (c_cursor_array,
				   i,
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str ("x"))),
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str ("y"))),
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str
						("pot-offset"))),
				   &(c_game_state->game_struct->map),
				   c_game_state->game_struct->options.
				   max_cursor_pot_offset);
	}
      lw6ker_cursor_array_force_nb_cursors (c_cursor_array, n);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6ker_enable_bot (SCM game_state, SCM team_id)
{
  SCM ret = SCM_BOOL_F;
  LW6KER_GAME_STATE *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-enable-bot");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-enable-bot");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      ret =
	lw6ker_game_state_enable_bot (c_game_state,
				      c_team_id) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6ker_disable_bot (SCM game_state, SCM team_id)
{
  SCM ret = SCM_BOOL_F;
  LW6KER_GAME_STATE *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-disable-bot");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-disable-bot");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      ret =
	lw6ker_game_state_disable_bot (c_game_state,
				       c_team_id) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6ker_do_round (SCM game_state)
{
  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-do-round");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  lw6ker_game_state_do_round (c_game_state);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6ker_get_moves (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-moves");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_moves (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_get_spreads (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-spreads");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_spreads (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_get_rounds (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-rounds");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_rounds (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_add_team (SCM game_state, SCM team_id, SCM nb_cursors)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      int c_team_id;
      int c_nb_cursors;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-add-team");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
		  "c-lw6ker-add-team");
      SCM_ASSERT (scm_is_integer (nb_cursors), nb_cursors,
		  SCM_ARG3, "c-lw6ker-add-team");
      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      c_nb_cursors = scm_to_int (nb_cursors);
      if (c_game_state)
	{
	  ret =
	    lw6ker_game_state_add_team (c_game_state,
					c_team_id,
					c_nb_cursors) ?
	    SCM_BOOL_T : SCM_BOOL_F;
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

static SCM
scm_lw6ker_remove_team (SCM game_state, SCM team_id, SCM nb_cursors)
{
  SCM ret;
  if (lw6_global.gfx_backend != NULL)
    {
      LW6KER_GAME_STATE *c_game_state;
      int c_team_id;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-remove-team");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
		  "c-lw6ker-remove-team");
      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      if (c_game_state)
	{
	  ret =
	    lw6ker_game_state_remove_team (c_game_state,
					   c_team_id) ?
	    SCM_BOOL_T : SCM_BOOL_F;
	}
      else
	{
	  ret = SCM_UNDEFINED;
	}
    }
  else
    {
      ret = SCM_UNDEFINED;
    }

  return ret;
}

/*
 * In liquidwar6snd
 */

/*
 * In backend.c
 */
static SCM
scm_lw6snd_get_backends ()
{
  SCM ret = SCM_UNDEFINED;
  LW6SYS_ASSOC *backends;
  LW6SYS_LIST *keys;
  LW6SYS_LIST *key;
  char *module_id;
  char *module_name;

  backends = lw6snd_get_backends ();
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  return ret;
}

static SCM
scm_lw6snd_create_backend (SCM backend_name)
{
  SCM ret;

  if (lw6_global.snd_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, "c-lw6snd-create-backend");

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.snd_backend = lw6snd_create_backend (c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.snd_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6snd_destroy_backend ()
{
  if (lw6_global.snd_backend != NULL)
    {
      if (lw6_global.snd_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "",
		      _
		      ("destroying snd backend, but backend has not been previously cleaned up"));
	  lw6snd_quit (lw6_global.snd_backend);
	}

      lw6snd_destroy_backend (lw6_global.snd_backend);
    }
  lw6_global.snd_backend = NULL;
  lw6_global.snd_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
scm_lw6snd_init (SCM sound_volume, SCM music_volume)
{
  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1,
	      "c-lw6snd-init");
  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG2,
	      "c-lw6snd-init");

  if (lw6_global.snd_backend != NULL)
    {
      float c_sound_volume;
      float c_music_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      c_music_volume = scm_num2float (music_volume, 0, NULL);

      lw6snd_init (lw6_global.snd_backend, c_sound_volume, c_music_volume);
      lw6_global.snd_initialized = 1;
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_quit ()
{
  if (lw6_global.snd_backend != NULL)
    {
      lw6snd_quit (lw6_global.snd_backend);
    }
  lw6_global.snd_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_play_sound (SCM sound_id)
{
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_integer (sound_id), sound_id, SCM_ARG1,
	      "c-lw6snd-play-sound");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      int c_sound_id;

      c_sound_id = scm_to_int (sound_id);

      ret =
	scm_int2num (lw6snd_play_sound (lw6_global.snd_backend, c_sound_id));
    }

  return ret;
}

static SCM
scm_lw6snd_set_sound_volume (SCM sound_volume)
{
  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1,
	      "c-lw6snd-set-sound-volume");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_sound_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      lw6snd_set_sound_volume (lw6_global.snd_backend, c_sound_volume);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_play_music (SCM music_name)
{
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (music_name), music_name, SCM_ARG1,
	      "c-lw6snd-play-music");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      char *c_music_name;

      c_music_name = to_0str (music_name);
      if (c_music_name)
	{
	  ret =
	    scm_int2num (lw6snd_play_music
			 (lw6_global.snd_backend, c_music_name));
	  LW6SYS_FREE (c_music_name);
	}
    }

  return ret;
}

static SCM
scm_lw6snd_set_music_volume (SCM music_volume)
{
  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG1,
	      "c-lw6snd-set-music-volume");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_music_volume;

      c_music_volume = scm_num2float (music_volume, 0, NULL);
      lw6snd_set_music_volume (lw6_global.snd_backend, c_music_volume);
    }

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6con
 */

/*
 * In backend.c
 */
static SCM
scm_lw6con_init ()
{
  lw6con_handler_install (lw6_con_handler);
  lw6_global.con_initialized = 1;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6con_quit ()
{
  lw6con_handler_remove ();
  lw6_global.con_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6con_poll ()
{
  lw6con_handler_poll ();

  return SCM_UNDEFINED;
}


/*
 * In liquidwar6net
 */

/*
 * In setup.c
 */
static SCM
scm_lw6net_init ()
{
  SCM ret = SCM_BOOL_F;

  lw6_global.net_context = lw6net_init ();
  if (lw6_global.net_context)
    {
      lw6_global.net_initialized = 1;
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6net_quit ()
{
  if (lw6_global.net_initialized && (lw6_global.net_context != NULL))
    {
      lw6net_quit (lw6_global.net_context);
    }
  lw6_global.net_context = NULL;
  lw6_global.net_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6net_server_start (SCM ip, SCM port)
{
  SCM ret = SCM_BOOL_F;
  char *c_ip;
  int c_port;

  SCM_ASSERT (scm_is_string (ip), ip, SCM_ARG1, "c-lw6net-server-start");
  SCM_ASSERT (scm_is_integer (port), port, SCM_ARG2, "c-lw6net-server-start");

  if (lw6_global.net_context)
    {
      c_ip = to_0str (ip);
      if (c_ip)
	{
	  c_port = scm_to_int (port);
	  ret =
	    lw6net_server_start (lw6_global.net_context, c_ip,
				 c_port) ? SCM_BOOL_T : SCM_BOOL_F;
	  LW6SYS_FREE (c_ip);
	}
    }

  return ret;
}

static SCM
scm_lw6net_server_poll_idle ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.net_context)
    {
      ret =
	lw6net_server_poll (lw6_global.net_context,
			    NULL, NULL) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6net_server_poll_playing (SCM map, SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.net_context)
    {
      LW6MAP_MAP *c_map;
      LW6KER_GAME_STATE *c_game_state;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map,
		   map), map, SCM_ARG1, "c-lw6net-server-poll");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2, "c-lw6net-server-poll");

      c_map = lw6_scm_to_map (map);
      c_game_state = lw6_scm_to_game_state (game_state);

      ret =
	lw6net_server_poll (lw6_global.net_context,
			    c_map, c_game_state) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6net_server_stop ()
{
  if (lw6_global.net_context)
    {
      lw6net_server_stop (lw6_global.net_context);
    }

  return SCM_UNDEFINED;
}

/*
 * Register all the functions, make them callable from Guile.
 */
int
lw6_register_funcs ()
{
  int ret = 1;
  scm_c_define_gsubr ("c-gettext", 1, 0, 0, (SCM (*)())scm_gettext);
  scm_c_define_gsubr ("_", 1, 0, 0, (SCM (*)())scm_gettext);
  /*
   * In liquidwar6sys
   */
  /*
   * In log.c
   */
  scm_c_define_gsubr ("c-lw6sys-log", 2, 0, 0, (SCM (*)())scm_lw6sys_log);
  /*
   * In time.c
   */
  scm_c_define_gsubr ("c-lw6sys-timestamp",
		      0, 0, 0, (SCM (*)())scm_lw6sys_timestamp);
  /*
   * In liquidwar6cfg
   */
  /*
   * In commandline.c
   */
  scm_c_define_gsubr ("c-lw6cfg-parse-command-line",
		      0, 0, 0, (SCM (*)())scm_lw6cfg_parse_command_line);
  /*
   * In dir.c
   */
  scm_c_define_gsubr
    ("c-lw6cfg-get-default-system-config-file", 0, 0,
     0, (SCM (*)())scm_lw6cfg_get_default_system_config_file);
  scm_c_define_gsubr
    ("c-lw6cfg-get-default-user-config-file", 0, 0, 0,
     (SCM (*)())scm_lw6cfg_get_default_user_config_file);
  /*
   * In load.c
   */
  scm_c_define_gsubr ("c-lw6cfg-load", 1, 0, 0, (SCM (*)())scm_lw6cfg_load);
  /*
   * In option.c
   */
  scm_c_define_gsubr ("c-lw6cfg-option-exists?",
		      1, 0, 0, (SCM (*)())scm_lw6cfg_option_exists);
  scm_c_define_gsubr ("c-lw6cfg-get-option", 1, 0, 0,
		      (SCM (*)())scm_lw6cfg_get_option);
  scm_c_define_gsubr ("c-lw6cfg-set-option", 2, 0, 0,
		      (SCM (*)())scm_lw6cfg_set_option);
  /*
   * In save.c
   */
  scm_c_define_gsubr ("c-lw6cfg-save", 1, 0, 0, (SCM (*)())scm_lw6cfg_save);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6cfg-init", 0, 0, 0, (SCM (*)())scm_lw6cfg_init);
  scm_c_define_gsubr ("c-lw6cfg-quit", 0, 0, 0, (SCM (*)())scm_lw6cfg_quit);

  /*
   * In liquidwar6opt
   */
  /*
   * In static.c
   */
  scm_c_define_gsubr ("c-lw6opt-static-get-options-list", 0, 0, 0,
		      (SCM (*)())scm_lw6opt_static_get_options_list);
  scm_c_define_gsubr ("c-lw6opt-static-get-int", 2, 0, 0,
		      (SCM (*)())scm_lw6opt_static_get_int);

  /*
   * In liquidwar6gfx
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6gfx-get-backends", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_get_backends);
  scm_c_define_gsubr ("c-lw6gfx-create-backend",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_create_backend);
  scm_c_define_gsubr ("c-lw6gfx-destroy-backend", 0,
		      0, 0, (SCM (*)())scm_lw6gfx_destroy_backend);
  /*
   * In background.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-background",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_display_background);
  /*
   * In event.c
   */
  scm_c_define_gsubr ("c-lw6gfx-poll-keypress",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_keypress);
  scm_c_define_gsubr ("c-lw6gfx-is-key-esc", 1, 0, 0,
		      (SCM (*)())scm_lw6gfx_is_key_esc);
  scm_c_define_gsubr ("c-lw6gfx-is-key-up", 1, 0, 0,
		      (SCM (*)())scm_lw6gfx_is_key_up);
  scm_c_define_gsubr ("c-lw6gfx-is-key-down", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_down);
  scm_c_define_gsubr ("c-lw6gfx-is-key-left", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_left);
  scm_c_define_gsubr ("c-lw6gfx-is-key-right", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_right);
  scm_c_define_gsubr ("c-lw6gfx-is-key-enter", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_enter);
  scm_c_define_gsubr ("c-lw6gfx-is-key-del", 1, 0, 0,
		      (SCM (*)())scm_lw6gfx_is_key_del);
  scm_c_define_gsubr ("c-lw6gfx-is-key-backspace", 1,
		      0, 0, (SCM (*)())scm_lw6gfx_is_key_backspace);
  scm_c_define_gsubr ("c-lw6gfx-is-key-help", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_help);
  scm_c_define_gsubr ("c-lw6gfx-is-key-quit", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_quit);
  scm_c_define_gsubr ("c-lw6gfx-poll-quit", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_poll_quit);
  scm_c_define_gsubr ("c-lw6gfx-send-quit", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_send_quit);
  scm_c_define_gsubr ("c-lw6gfx-get-key-state", 1,
		      0, 0, (SCM (*)())scm_lw6gfx_get_key_state);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-move",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_move);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-button",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_button);
  scm_c_define_gsubr ("c-lw6gfx-get-mouse-state", 0,
		      0, 0, (SCM (*)())scm_lw6gfx_get_mouse_state);

  /*
   * In hud.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-hud",
		      3, 0, 0, (SCM (*)())scm_lw6gfx_display_hud);
  scm_c_define_gsubr ("c-lw6gfx-display-hud-preview",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_hud_preview);

  /*
   * In info.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-sysinfo",
		      4, 0, 0, (SCM (*)())scm_lw6gfx_display_sysinfo);
  scm_c_define_gsubr ("c-lw6gfx-display-log", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_display_log);
  scm_c_define_gsubr ("c-lw6gfx-display-time", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_display_time);
  scm_c_define_gsubr ("c-lw6gfx-display-version", 0,
		      0, 0, (SCM (*)())scm_lw6gfx_display_version);
  /*
   * In map.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-map-preview",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_map_preview);
  scm_c_define_gsubr ("c-lw6gfx-display-map-zones",
		      3, 0, 0, (SCM (*)())scm_lw6gfx_display_map_zones);
  scm_c_define_gsubr
    ("c-lw6gfx-display-map-gradient", 4, 0, 0,
     (SCM (*)())scm_lw6gfx_display_map_gradient);
  scm_c_define_gsubr ("c-lw6gfx-display-game", 4, 0,
		      0, (SCM (*)())scm_lw6gfx_display_game);
  scm_c_define_gsubr ("c-lw6gfx-pick-map-coord", 5, 0,
		      0, (SCM (*)())scm_lw6gfx_pick_map_coord);
  /*
   * In menu.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-menu",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_menu);
  scm_c_define_gsubr ("c-lw6gfx-pick-menuitem",
		      4, 0, 0, (SCM (*)())scm_lw6gfx_pick_menuitem);

  /* 
   * In render.c
   */
  scm_c_define_gsubr ("c-lw6gfx-prepare-buffer",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_prepare_buffer);
  scm_c_define_gsubr ("c-lw6gfx-swap-buffers", 0, 0,
		      0, (SCM (*)())scm_lw6gfx_swap_buffers);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6gfx-init", 4, 0, 0, (SCM (*)())scm_lw6gfx_init);
  scm_c_define_gsubr ("c-lw6gfx-set-video-mode", 3,
		      0, 0, (SCM (*)())scm_lw6gfx_set_video_mode);
  scm_c_define_gsubr ("c-lw6gfx-quit", 0, 0, 0, (SCM (*)())scm_lw6gfx_quit);
  /*
   * In timer.c
   */
  scm_c_define_gsubr ("c-lw6gfx-delay", 1, 0, 0, (SCM (*)())scm_lw6gfx_delay);
  scm_c_define_gsubr ("c-lw6gfx-get-ticks", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_get_ticks);
  /*
   * In gamelook.c
   */
  scm_c_define_gsubr ("c-lw6gfx-default-game-look",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_default_game_look);
  /*
   * In liquidwar6map
   */
  /*
   * In read.c
   */
  scm_c_define_gsubr ("c-lw6map-get-system-dir",
		      0, 0, 0, (SCM (*)())scm_lw6map_get_system_dir);
  scm_c_define_gsubr ("c-lw6map-get-maps", 1, 0, 0,
		      (SCM (*)())scm_lw6map_get_maps);
  scm_c_define_gsubr ("c-lw6map-read", 1, 0, 0, (SCM (*)())scm_lw6map_read);
  scm_c_define_gsubr
    ("c-lw6map-get-background-style", 1, 0, 0,
     (SCM (*)())scm_lw6map_get_background_style);
  scm_c_define_gsubr ("c-lw6map-get-hud-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_hud_style);
  scm_c_define_gsubr ("c-lw6map-get-menu-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_menu_style);
  scm_c_define_gsubr ("c-lw6map-get-view-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_view_style);
  /*
   * In liquidwar6ker
   */
  scm_c_define_gsubr ("c-lw6ker-build-game-struct",
		      1, 0, 0, (SCM (*)())scm_lw6ker_build_game_struct);
  scm_c_define_gsubr ("c-lw6ker-build-game-state", 1,
		      0, 0, (SCM (*)())scm_lw6ker_build_game_state);
  scm_c_define_gsubr ("c-lw6ker-copy-game-state", 2,
		      0, 0, (SCM (*)())scm_lw6ker_copy_game_state);
  scm_c_define_gsubr
    ("c-lw6ker-game-struct-checksum", 1, 0, 0,
     (SCM (*)())scm_lw6ker_game_struct_checksum);
  scm_c_define_gsubr ("c-lw6ker-game-state-checksum",
		      1, 0, 0, (SCM (*)())scm_lw6ker_game_state_checksum);
  scm_c_define_gsubr ("c-lw6ker-get-cursors",
		      2, 0, 0, (SCM (*)())scm_lw6ker_get_cursors);
  scm_c_define_gsubr ("c-lw6ker-set-cursors",
		      3, 0, 0, (SCM (*)())scm_lw6ker_set_cursors);
  scm_c_define_gsubr ("c-lw6ker-enable-bot",
		      2, 0, 0, (SCM (*)())scm_lw6ker_enable_bot);
  scm_c_define_gsubr ("c-lw6ker-disable-bot",
		      2, 0, 0, (SCM (*)())scm_lw6ker_disable_bot);
  scm_c_define_gsubr ("c-lw6ker-do-round",
		      1, 0, 0, (SCM (*)())scm_lw6ker_do_round);
  scm_c_define_gsubr ("c-lw6ker-get-moves", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_moves);
  scm_c_define_gsubr ("c-lw6ker-get-spreads", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_spreads);
  scm_c_define_gsubr ("c-lw6ker-get-rounds", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_rounds);
  scm_c_define_gsubr ("c-lw6ker-add-team", 3, 0, 0,
		      (SCM (*)())scm_lw6ker_add_team);
  scm_c_define_gsubr ("c-lw6ker-remove-team", 2, 0,
		      0, (SCM (*)())scm_lw6ker_remove_team);
  /*
   * In liquidwar6snd
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6snd-get-backends", 0, 0, 0,
		      (SCM (*)())scm_lw6snd_get_backends);
  scm_c_define_gsubr ("c-lw6snd-create-backend",
		      1, 0, 0, (SCM (*)())scm_lw6snd_create_backend);
  scm_c_define_gsubr ("c-lw6snd-destroy-backend", 0,
		      0, 0, (SCM (*)())scm_lw6snd_destroy_backend);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6snd-init", 2, 0, 0, (SCM (*)())scm_lw6snd_init);
  scm_c_define_gsubr ("c-lw6snd-quit", 0, 0, 0, (SCM (*)())scm_lw6snd_quit);

  /*
   * In sound.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-sound", 1,
		      0, 0, (SCM (*)())scm_lw6snd_play_sound);
  scm_c_define_gsubr ("c-lw6snd-set-sound-volume", 1,
		      0, 0, (SCM (*)())scm_lw6snd_set_sound_volume);

  /*
   * In music.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-music", 1,
		      0, 0, (SCM (*)())scm_lw6snd_play_music);
  scm_c_define_gsubr ("c-lw6snd-set-music-volume", 1,
		      0, 0, (SCM (*)())scm_lw6snd_set_music_volume);

  /*
   * In liquidwar6con
   */
  /*
   * In handler.c
   */
  scm_c_define_gsubr ("c-lw6con-init", 0, 0, 0, (SCM (*)())scm_lw6con_init);
  scm_c_define_gsubr ("c-lw6con-quit", 0, 0, 0, (SCM (*)())scm_lw6con_quit);
  scm_c_define_gsubr ("c-lw6con-poll", 0, 0, 0, (SCM (*)())scm_lw6con_poll);

  /*
   * In liquidwar6net
   */
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6net-init", 0, 0, 0, (SCM (*)())scm_lw6net_init);
  scm_c_define_gsubr ("c-lw6net-quit", 0, 0, 0, (SCM (*)())scm_lw6net_quit);
  scm_c_define_gsubr ("c-lw6net-server-start", 2, 0, 0,
		      (SCM (*)())scm_lw6net_server_start);
  scm_c_define_gsubr ("c-lw6net-server-poll-idle", 0, 0, 0,
		      (SCM (*)())scm_lw6net_server_poll_idle);
  scm_c_define_gsubr ("c-lw6net-server-poll-playing", 2, 0, 0,
		      (SCM (*)())scm_lw6net_server_poll_playing);
  scm_c_define_gsubr ("c-lw6net-server-stop", 0, 0, 0,
		      (SCM (*)())scm_lw6net_server_stop);

  return ret;
}

extern void
lw6_con_handler (char *c_line)
{
  SCM line;
  SCM func_symbol;
  SCM func;

  if (c_line)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "", _("interpreting console input \"%s\""),
		  c_line);

      lw6con_history_add_if_needed (c_line);
      line = scm_makfrom0str (c_line);

      func_symbol = scm_c_lookup ("lw6-console-try-catch");
      func = scm_variable_ref (func_symbol);
      scm_call_1 (func, line);
      free (c_line);		// exceptionnally, don't use LW6SYS_FREE
    }
  else
    {
      lw6_exit ();
    }
}
